package api

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/labstack/echo/v4"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	netutil "a.yandex-team.ru/infra/walle/server/go/internal/lib/net"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
)

type HandleAgentReportSuite struct {
	suite.Suite
	store    *MockStore
	hosts    map[string]*host
	hostName string
	knownMAC netutil.MAC
	calls    map[string]int
}

func (suite *HandleAgentReportSuite) SetupSuite() {
	suite.hostName = "host.fqdn"
	suite.knownMAC = "94:de:80:8c:db:46"
	suite.hosts = map[string]*host{suite.hostName: {macs: []netutil.MAC{suite.knownMAC}}}
	suite.store = &MockStore{}
	suite.store.On("GetDevStands", mock.Anything).Return(nil)
	suite.store.On("SaveHostMACs", mock.Anything, mock.Anything).Return()
	suite.store.On("GetHost", mock.Anything, mock.Anything).Return(
		func(ctx context.Context, name string) *host { return suite.hosts[name] },
		func(ctx context.Context, name string) error {
			if suite.hosts[name] == nil {
				return errHostNotFound
			}
			return nil
		})
	suite.store.On("GetOrCreateNetwork", mock.Anything, mock.Anything).Return(&repos.HostNetwork{}, nil)
}

func (suite *HandleAgentReportSuite) push(hostName string, report *agentReport) *httptest.ResponseRecorder {
	b, err := json.Marshal(report)
	suite.Require().NoError(err)
	e := echo.New()
	req := httptest.NewRequest(http.MethodPut, "/v1/hosts", bytes.NewReader(b))
	req.Header.Set(echo.HeaderContentType, echo.MIMEApplicationJSON)
	rec := httptest.NewRecorder()
	c := e.NewContext(req, rec)
	c.SetPath("/:host_name/agent-report")
	c.SetParamNames("host_name")
	c.SetParamValues(hostName)
	_ = handle(c, suite.store)
	return rec
}

func (suite *HandleAgentReportSuite) setStoreCalls() {
	methods := map[string][]interface{}{
		"SaveActiveMAC":    {mock.Anything, mock.Anything, mock.Anything},
		"SaveIPs":          {mock.Anything, mock.Anything, mock.Anything},
		"SaveSwitches":     {mock.Anything, mock.Anything, mock.Anything},
		"SaveAgentVersion": {mock.Anything, mock.Anything},
		"SaveAgentErrors":  {mock.Anything, mock.Anything},
	}
	suite.calls = make(map[string]int)
	for method, args := range methods {
		method := method
		suite.store.On(method, args...).Run(func(mock.Arguments) { suite.calls[method]++ }).Return()
	}
}

func (suite *HandleAgentReportSuite) TestValidate() {
	report := &agentReport{
		Version:  "1.0.0",
		Switches: []*networkSwitch{{Name: "sas2-112s153"}},
	}
	resp := suite.push(suite.hostName, report)
	suite.Require().Equal(http.StatusBadRequest, resp.Code)
}

func (suite *HandleAgentReportSuite) TestFullRequest() {
	suite.setStoreCalls()
	report := &agentReport{
		Version:  "1.0.0",
		MACs:     map[netutil.MAC]bool{suite.knownMAC: true, "94:de:80:8c:db:47": false},
		IPs:      []netutil.IP{"2a02:6b8:c02:f0f:0:510:a107:ec8a", "2a02:6b8:fc02:90f:0:510:a107:ec8a"},
		Switches: []*networkSwitch{{Name: "sas2-112s153", Port: "10GE1/0/10", Time: 1655982139}},
		Errors:   []string{"Failed to determine host's switch/port"},
	}
	rawResp := suite.push(suite.hostName, report)
	suite.Require().Equal(http.StatusOK, rawResp.Code)
	resp := &response{}
	suite.Require().NoError(json.Unmarshal(rawResp.Body.Bytes(), resp))
	suite.Require().Equal(&response{Result: successMessage, OtherStands: nil}, resp)
	for method, number := range suite.calls {
		suite.Assert().Equal(1, number, method)
	}
}

func (suite *HandleAgentReportSuite) TestUnknownActiveMAC() {
	suite.setStoreCalls()
	report := &agentReport{
		Version: "1.0.0",
		MACs:    map[netutil.MAC]bool{"unknown-mac": true},
	}
	rawResp := suite.push(suite.hostName, report)
	suite.Require().Equal(http.StatusOK, rawResp.Code)
	resp := &response{}
	suite.Require().NoError(json.Unmarshal(rawResp.Body.Bytes(), resp))
	suite.Require().Equal(
		&response{Result: fmt.Sprintf("Ignore report: %s", errUnknownActiveMAC.Error()), OtherStands: nil},
		resp,
	)
	for method, number := range suite.calls {
		suite.Assert().Equal(0, number, method)
	}
}

func (suite *HandleAgentReportSuite) TestUnknownHost() {
	suite.setStoreCalls()
	report := &agentReport{
		Version: "1.0.0",
	}
	rawResp := suite.push("unknown.host", report)
	suite.Require().Equal(http.StatusOK, rawResp.Code)
	resp := &response{}
	suite.Require().NoError(json.Unmarshal(rawResp.Body.Bytes(), resp))
	suite.Require().Equal(
		&response{Result: fmt.Sprintf("Ignore report: %s", errHostNotFound.Error()), OtherStands: nil},
		resp,
	)
	for method, number := range suite.calls {
		suite.Assert().Equal(0, number, method)
	}
}

func TestHandleJugglerChecks(t *testing.T) {
	suite.Run(t, new(HandleAgentReportSuite))
}
