package api

import (
	"context"
	"encoding/hex"
	"errors"
	"fmt"
	"sort"
	"sync"
	"time"

	"github.com/gofrs/uuid"
	"go.mongodb.org/mongo-driver/mongo"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/automation"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/client"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	netutil "a.yandex-team.ru/infra/walle/server/go/internal/lib/net"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/notification"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	"a.yandex-team.ru/library/go/core/log"
)

const (
	sourceAgent = "agent"
	sourceLLDP  = "lldp"

	auditLogIssuer = "wall-e"

	agentMessagesModule = "agent"
)

var errHostNotFound = errors.New("host not found")

type HostRepo interface {
	SelectOne(ctx context.Context, name string, keys []string) (*db.MongoSelection, error)
	NewBulkWriter(options *db.MongoBulkWriterOptions) (db.BulkWriter, error)
}

type NetworkRepo interface {
	GetOrCreate(ctx context.Context, uuid string) (*repos.HostNetwork, error)
	NewBulkWriter(options *db.MongoBulkWriterOptions) (db.BulkWriter, error)
}

type HostMacsRepo interface {
	NewBulkWriter(options *db.MongoBulkWriterOptions) (db.BulkWriter, error)
}

type AuditLogRepo interface {
	Insert(ctx context.Context, record *repos.AuditLog) error
}

type CacheRepo interface {
	GetStandHosts(ctx context.Context, stand string) ([]string, int64, error)
}

type store struct {
	hostRepo       HostRepo
	networkRepo    NetworkRepo
	auditLogRepo   AuditLogRepo
	bulkWriters    *bulkWriters
	racktables     client.Racktables
	logger         log.Logger
	devStandsStore *devStandsStore
}

type bulkWriters struct {
	host     db.BulkWriter
	network  db.BulkWriter
	hostMACs db.BulkWriter
}

func newStore(
	logger log.Logger,
	hostRepo HostRepo,
	networkRepo NetworkRepo,
	macsRepo HostMacsRepo,
	auditLogRepo AuditLogRepo,
	cacheRepo CacheRepo,
	racktables client.Racktables,
	devStands map[string]string,
) (*store, error) {
	writers := &bulkWriters{}
	opts := &db.MongoBulkWriterOptions{
		RPSLimit:      5,
		FlushInterval: 2 * time.Second,
		Size:          1000,
		ResultHandler: func(res *mongo.BulkWriteResult, err error) {
			if err != nil {
				logger.Errorf("Failed to write to DB: %v", err)
			}
		},
		ErrorCollector: func(err error) {
			logger.Errorf("Error in bulk writer: %v", err)
		},
	}
	var err error
	if writers.host, err = hostRepo.NewBulkWriter(opts); err != nil {
		return nil, err
	}
	if writers.network, err = networkRepo.NewBulkWriter(opts); err != nil {
		return nil, err
	}
	if writers.hostMACs, err = macsRepo.NewBulkWriter(opts); err != nil {
		return nil, err
	}
	return &store{
		hostRepo:     hostRepo,
		networkRepo:  networkRepo,
		auditLogRepo: auditLogRepo,
		bulkWriters:  writers,
		racktables:   racktables,
		logger:       logger,
		devStandsStore: &devStandsStore{
			logger:     logger,
			standURLs:  devStands,
			cacheRepo:  cacheRepo,
			localCache: make(map[string][]string),
		},
	}, nil
}

func (s *store) GetDevStands(hostName string) []string {
	return s.devStandsStore.get(hostName)
}

func (s *store) GetHost(ctx context.Context, hostName string) (*host, error) {
	selection, err := s.hostRepo.SelectOne(
		ctx,
		hostName,
		[]string{
			repos.HostFieldKeyID,
			repos.HostFieldKeyInv,
			repos.HostFieldKeyProject,
			repos.HostFieldKeyMACs,
			repos.HostFieldKeyIPs,
			repos.HostFieldKeyLocation,
			repos.HostFieldKeyState,
			repos.HostFieldKeyAgentVersion,
		},
	)
	if err != nil {
		return nil, err
	}
	defer selection.Close()
	if selection.Next() {
		res := &host{name: hostName}
		if err = selection.Scan(
			&res.uuid, &res.inv, &res.project, &res.macs, &res.ips, &res.location, &res.state, &res.agentVersion,
		); err != nil {
			return nil, err
		}
		if res.state != automation.HostStatusInvalid {
			return res, nil
		}
	}
	return nil, errHostNotFound
}

func (s *store) GetOrCreateNetwork(ctx context.Context, hostUUID string) (*repos.HostNetwork, error) {
	return s.networkRepo.GetOrCreate(ctx, hostUUID)
}

func (s *store) SaveHostMACs(report *agentReport, hostName string) {
	id := repos.NewHostMacsID(hostName, report.macList)
	timestamp := report.timestamp.Unix()
	s.bulkWriters.hostMACs.Upsert(&db.BulkWriterElem{
		ID:     id,
		Fields: map[string]interface{}{repos.HostMacsFieldKeyLastTime: timestamp},
		SetOnInsert: &repos.HostMacs{
			ID:        id,
			FirstTime: timestamp,
			MACs:      report.macList,
			Name:      hostName,
		},
	})
}

func (s *store) SaveActiveMAC(report *agentReport, host *host, network *repos.HostNetwork) {
	switch {
	case len(report.activeMACs) != 1:
		s.logger.Warnf("host %s reported no active MACs or more than one: %v", host.name, report.activeMACs)
	default:
		if report.timestamp.Unix() <= network.ActiveMacTime {
			return
		}
		s.bulkWriters.network.Update(&db.BulkWriterElem{
			ID: network.HostUUID,
			Fields: map[string]interface{}{
				repos.HostNetworkFieldKeyActiveMacTime:   report.timestamp.Unix(),
				repos.HostNetworkFieldKeyActiveMacSource: sourceAgent,
				repos.HostNetworkFieldKeyActiveMac:       report.activeMACs[0],
			},
		})
		s.bulkWriters.host.Update(&db.BulkWriterElem{
			ID: host.uuid,
			Fields: map[string]interface{}{
				repos.HostFieldKeyActiveMacSource: sourceAgent,
				repos.HostFieldKeyActiveMac:       report.activeMACs[0],
			},
		})

		if report.activeMACs[0] != network.ActiveMac {
			payload := map[string]interface{}{
				"new": &auditLogPayloadRecordOnMacChange{
					Mac:    report.activeMACs[0],
					Time:   report.timestamp.Unix(),
					Source: sourceAgent,
				},
				"prev": &auditLogPayloadRecordOnMacChange{
					Mac:    network.ActiveMac,
					Time:   network.ActiveMacTime,
					Source: network.ActiveMacSource,
				},
			}
			if err := s.createAuditLogRecord(repos.AuditLogTypeActiveMacChanged, report, host, payload); err != nil {
				s.logger.Errorf("create audit log record for %s host: %v", host.name, err)
				break
			}
		}
	}
}

func (s *store) SaveIPs(report *agentReport, host *host, network *repos.HostNetwork) {
	sort.Slice(report.IPs, func(i, j int) bool { return report.IPs[i] < report.IPs[j] })
	if !netutil.IsSliceIPEqual(report.IPs, host.ips) {
		s.bulkWriters.host.Update(&db.BulkWriterElem{
			ID:     host.uuid,
			Fields: map[string]interface{}{repos.HostFieldKeyIPs: report.IPs},
		})
		if err := s.createAuditLogRecord(repos.AuditLogTypeIPsChanged, report, host, map[string]interface{}{
			"old": host.ips,
			"new": report.IPs,
		}); err != nil {
			s.logger.Errorf("create audit log record for %s host: %v", host.name, err)
		}
	}
	s.bulkWriters.network.Update(&db.BulkWriterElem{
		ID: network.HostUUID,
		Fields: map[string]interface{}{
			repos.HostNetworkFieldKeyIPs:     report.IPs,
			repos.HostNetworkFieldKeyIPsTime: report.timestamp.Unix(),
		},
	})
}

func (s *store) SaveSwitches(report *agentReport, host *host, network *repos.HostNetwork) {
	switch {
	case len(report.Switches) > 1:
		s.logger.Warnf("host %s reported more than one switch/port: %+v", host.name, report.Switches)
	case len(report.Switches) == 1:
		if network.NetworkTime > report.Switches[0].Time {
			break
		}
		switchName := netutil.ShortenRacktablesSwitchName(report.Switches[0].Name)
		port := netutil.ShortenRacktablesPortName(report.Switches[0].Port)

		// WALLE-3855 Do not set switch and port to 'unknown' when LLDP on host fails to get them.
		if switchName == "unknown" || port == "unknown" {
			break
		}

		switchPortExist, err := s.racktables.HasSwitchPort(switchName, port)
		if err != nil {
			s.logger.Errorf("Failed to check switch/port in Racktables: %v", err)
			break
		}
		if !switchPortExist {
			s.logger.Warnf(
				"host %s reported a possibly invalid switch/port: %s does not have %s (%s) port",
				host.name,
				report.Switches[0].Name,
				report.Switches[0].Port,
				port,
			)
			break
		}

		s.bulkWriters.network.Update(&db.BulkWriterElem{
			ID: network.HostUUID,
			Fields: map[string]interface{}{
				repos.HostNetworkFieldKeySwitch:        switchName,
				repos.HostNetworkFieldKeyPort:          port,
				repos.HostNetworkFieldKeyNetworkSource: sourceLLDP,
				repos.HostNetworkFieldKeyNetworkTime:   report.Switches[0].Time,
			},
		})

		s.bulkWriters.host.Update(&db.BulkWriterElem{
			ID: host.uuid,
			Fields: map[string]interface{}{
				fmt.Sprintf("%s.%s", repos.HostFieldKeyLocation, repos.HostLocationFieldKeySwitch):        switchName,
				fmt.Sprintf("%s.%s", repos.HostFieldKeyLocation, repos.HostLocationFieldKeyPort):          port,
				fmt.Sprintf("%s.%s", repos.HostFieldKeyLocation, repos.HostLocationFieldKeyNetworkSource): sourceLLDP,
			},
		})

		if network.NetworkSwitch != switchName || network.NetworkPort != port {
			payload := map[string]interface{}{
				"new": &auditLogPayloadRecordOnSwitchPortChange{
					Switch: switchName,
					Port:   port,
					Source: sourceLLDP,
					Time:   report.Switches[0].Time,
				},
				"prev": &auditLogPayloadRecordOnSwitchPortChange{
					Switch: network.NetworkSwitch,
					Port:   network.NetworkPort,
					Source: network.NetworkSource,
					Time:   network.NetworkTime,
				},
			}
			if err := s.createAuditLogRecord(repos.AuditLogTypeSwitchPortChanged, report, host, payload); err != nil {
				s.logger.Errorf("create audit log record for %s host: %v", host.name, err)
				break
			}
		}
	}
}

func (s *store) SaveAgentVersion(report *agentReport, host *host) {
	if host.agentVersion != report.Version {
		s.bulkWriters.host.
			Update(&db.BulkWriterElem{
				ID:     host.uuid,
				Fields: map[string]interface{}{repos.HostFieldKeyAgentVersion: report.Version},
			})
	}
}

func (s *store) SaveAgentErrors(report *agentReport, host *host) {
	messages := make([]repos.HostMessage, len(report.Errors))
	for i, e := range report.Errors {
		s.logger.Infof("host %s reported error: %s", host.name, e)
		messages[i] = repos.HostMessageError(e)
	}
	s.bulkWriters.host.Update(&db.BulkWriterElem{
		ID:     host.uuid,
		Fields: map[string]interface{}{repos.HostFieldKeyAgentErrorFlag: len(report.Errors) > 0},
	})
	if len(report.Errors) > 0 {
		s.bulkWriters.host.Update(&db.BulkWriterElem{
			ID:     host.uuid,
			Fields: map[string]interface{}{fmt.Sprintf("%s.%s", repos.HostFieldKeyMessages, agentMessagesModule): messages},
		})
	} else {
		s.bulkWriters.host.Unset(&db.BulkWriterElem{
			ID:     host.uuid,
			Fields: map[string]interface{}{fmt.Sprintf("%s.%s", repos.HostFieldKeyMessages, agentMessagesModule): ""},
		})
	}
}

func (s *store) writeData(ctx context.Context) {
	writers := []db.BulkWriter{s.bulkWriters.host, s.bulkWriters.network, s.bulkWriters.hostMACs}
	for _, w := range writers {
		go w.Run()
	}
	<-ctx.Done()
	for _, w := range writers {
		w.Shutdown()
	}
}

func (s *store) createAuditLogRecord(
	logType repos.AuditLogType,
	report *agentReport,
	host *host,
	payload map[string]interface{},
) error {
	record := &repos.AuditLog{
		ID:       hex.EncodeToString(uuid.Must(uuid.NewV4()).Bytes()),
		Time:     float64(report.timestamp.Unix()),
		Status:   repos.AuditLogStatusCompleted,
		Issuer:   auditLogIssuer,
		Type:     logType,
		Project:  host.project,
		HostInv:  host.inv,
		HostUUID: host.uuid,
		HostName: host.name,
		Payload:  payload,
	}
	if err := s.auditLogRepo.Insert(report.ctx, record); err != nil {
		return err
	}
	return notification.SendMailOnEvent(record)
}

func (s *store) updateDevStands(ctx context.Context) {
	ticker := time.NewTicker(2 * time.Minute)
	defer ticker.Stop()
	for {
		if err := s.devStandsStore.refresh(ctx); err != nil {
			s.logger.Errorf("refresh development stands: %v", err)
		}
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
		}
	}
}

type devStandsStore struct {
	logger     log.Logger
	cacheRepo  CacheRepo
	mu         sync.RWMutex
	standURLs  map[string]string
	localCache map[string][]string
}

func (ds *devStandsStore) get(hostName string) []string {
	ds.mu.RLock()
	defer ds.mu.RUnlock()
	return ds.localCache[hostName]
}

func (ds *devStandsStore) refresh(ctx context.Context) error {
	tmp := make(map[string][]string)

	for stand, url := range ds.standURLs {
		hosts, timestamp, err := ds.cacheRepo.GetStandHosts(ctx, stand)
		if err != nil {
			return err
		}
		if time.Unix(timestamp, 0).Before(time.Now().Add(-time.Hour)) {
			ds.logger.Errorf("using an outdated list of hosts for '%s' dev. stand", stand)
		}
		for _, h := range hosts {
			tmp[h] = append(tmp[h], url)
		}
	}
	ds.mu.Lock()
	ds.localCache = tmp
	ds.mu.Unlock()
	return nil
}

type auditLogPayloadRecordOnMacChange struct {
	Mac    netutil.MAC `bson:"mac"`
	Time   int64       `bson:"actualization_time"`
	Source string      `bson:"source"`
}

type auditLogPayloadRecordOnSwitchPortChange struct {
	Switch netutil.SwitchName `bson:"switch"`
	Port   netutil.SwitchPort `bson:"port"`
	Source string             `bson:"source"`
	Time   int64              `bson:"actualization_time"`
}
