package api

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"runtime/debug"
	"time"

	"github.com/labstack/echo/v4"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/httputil"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
)

type Config struct {
	BaseApp     lib.AppConfig         `yaml:",inline"`
	MongoMain   db.MongoConfig        `yaml:"mongo"`
	MongoHealth db.MongoConfig        `yaml:"mongo_health"`
	HTTPServer  httputil.ServerConfig `yaml:"http_server"`
}

type application struct {
	config       *Config
	base         *lib.Application
	server       *http.Server
	store        *store
	mongoClients []*mongo.Client
}

func StartApplication(args *lib.CliArgs) {
	if err := os.Chdir(args.RootDir); err != nil {
		panic(err)
	}
	config := &Config{}
	if err := lib.LoadConfig(args.ConfigPath, config); err != nil {
		panic(err)
	}
	app, err := initApplication(config)
	if err != nil {
		panic(fmt.Errorf("init app: %v", err))
	}
	if err := app.run(); err != nil {
		app.base.Logger("base").Errorf("fatal: %v", err)
		panic(err)
	}
}

func initApplication(cnf *Config) (*application, error) {
	base, err := lib.NewApplication(cnf.BaseApp)
	if err != nil {
		return nil, err
	}
	if err = registerMetrics(); err != nil {
		return nil, err
	}
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	mainDB, err := db.NewMongoConnect(ctx, cnf.MongoMain)
	if err != nil {
		return nil, err
	}
	healthDB, err := db.NewMongoConnect(ctx, cnf.MongoHealth)
	if err != nil {
		return nil, err
	}
	cacheRepo := repos.NewCacheRepo(mainDB, db.NewMongoLocalReadPreference(cnf.MongoMain.GeoTags))
	healthRepo := repos.NewHealthRepo(healthDB, readpref.Primary())
	store, err := newStore(base, cacheRepo, healthRepo)
	if err != nil {
		return nil, err
	}

	e := echo.New()
	e.Logger.SetOutput(base.LoggerWriter("base"))
	e.Use(func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			defer func() {
				if err := recover(); err != nil {
					base.Logger("base").Errorf("recover from panic: %v\n%s", err, string(debug.Stack()))
				}
			}()
			return next(ctx)
		}
	})
	e.Use(httputil.ZapLoggerMiddleware(base.Logger("base")))
	e.POST(
		"/v1/health/push-statuses",
		func(ctx echo.Context) error {
			return handleJugglerChecks(ctx, store)
		},
	)
	e.GET("/metrics/v1/stats", monitoring.NewUnistatHTTPHandler())
	e.GET("/ping", monitoring.NewHealthCheckHTTPHandler())
	server, err := httputil.NewServer(cnf.HTTPServer, e)
	if err != nil {
		return nil, err
	}
	return &application{
		base:         base,
		config:       cnf,
		server:       server,
		store:        store,
		mongoClients: []*mongo.Client{mainDB.Client(), healthDB.Client()},
	}, nil
}

func (app *application) run() error {
	defer app.close()
	app.base.AddServeFunc("health-writer", app.store.write)
	app.base.AddServeFunc("rack-topologies", app.store.updateRackTopologies)
	app.base.AddServeFunc("http", httputil.NewServeFunc(app.config.HTTPServer, app.server))
	return app.base.Run()
}

func (app *application) close() {
	for _, item := range app.mongoClients {
		client := item
		func() {
			ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
			defer cancel()
			_ = client.Disconnect(ctx)
		}()
	}
}
