package api

import (
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/httputil"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
)

type HealthStore interface {
	HealthRepo
	QueueIsFull() bool
	TryPush([]*juggler.HostCheck) bool
	RackTopologyStore() juggler.RackTopologyStore
}

func handleJugglerChecks(ctx echo.Context, store HealthStore) error {

	defer monitoring.MeasureSecondsSince(metrics.ResponseTime, time.Now())
	if store.QueueIsFull() {
		metrics.Requests429.Update(1)
		return ctx.String(http.StatusTooManyRequests, "Requests queue is full")
	}

	request := &juggler.HealthRequest{}
	if err := ctx.Bind(&request); err != nil {
		metrics.Requests4xx.Update(1)
		return httputil.RespondError(ctx, http.StatusBadRequest, err, "parse request")
	}

	groupedChecks, checkKeys := juggler.ReshapeChecks(store.RackTopologyStore(), request, ctx.Logger())
	oldChecks, err := store.FindIDMap(ctx.Request().Context(), &repos.HostCheckFilter{Keys: checkKeys})
	if err != nil {
		metrics.Requests5xx.Update(1)
		return httputil.RespondError(ctx, http.StatusInternalServerError, err, "find old checks")
	}
	var convertedChecks []*juggler.HostCheck
	for _, children := range groupedChecks {
		convertedChecks = append(convertedChecks, juggler.ConvertChecks(children, oldChecks)...)
	}
	if !store.TryPush(convertedChecks) {
		metrics.Requests5xx.Update(1)
		return ctx.String(http.StatusTooManyRequests, "Requests queue is full")
	}
	metrics.Requests2xx.Update(1)
	return ctx.String(http.StatusAccepted, "")
}
