package api

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"sort"
	"strings"
	"testing"
	"time"

	"github.com/labstack/echo/v4"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	apimock "a.yandex-team.ru/infra/walle/server/go/internal/api/juggler_checks/mocks"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler"
	jugglermock "a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler/mocks"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
)

type HandleJugglerChecksSuite struct {
	suite.Suite
	store          *apimock.HealthStore
	receivedChecks map[juggler.WalleCheckKey]*juggler.HostCheck
	oldChecks      map[juggler.WalleCheckKey]*juggler.HostCheck
	rackTopologies map[string]map[string]int
	testRackName   string
}

func (suite *HandleJugglerChecksSuite) SetupSuite() {
	suite.testRackName = "wall-e.testing-man5-5A4"
	rackTopologyStore := &jugglermock.RackTopologyStore{}
	rackTopologyStore.
		On("Get", mock.Anything).
		Return(func(name string) map[string]int {
			return suite.rackTopologies[name]
		})
	suite.store = &apimock.HealthStore{}
	suite.store.On("QueueIsFull").Return(false)
	suite.store.On("TryPush", mock.Anything).Return(func(checks []*juggler.HostCheck) bool {
		for _, check := range checks {
			suite.receivedChecks[check.ID] = check
		}
		return true
	})
	suite.store.On("RackTopologyStore").Return(rackTopologyStore)
	suite.store.
		On("FindIDMap", mock.Anything, mock.Anything).
		Return(func(context.Context, *repos.HostCheckFilter) map[juggler.WalleCheckKey]*juggler.HostCheck {
			return suite.oldChecks
		}, func(context.Context, *repos.HostCheckFilter) error { return nil })

}

func (suite *HandleJugglerChecksSuite) SetupTest() {
	suite.receivedChecks = make(map[juggler.WalleCheckKey]*juggler.HostCheck)
	suite.oldChecks = make(map[juggler.WalleCheckKey]*juggler.HostCheck)
	suite.rackTopologies = make(map[string]map[string]int)

}

func (suite *HandleJugglerChecksSuite) push(request string) {
	e := echo.New()
	req := httptest.NewRequest(http.MethodPost, "/v1/health/push-statuses", strings.NewReader(request))
	req.Header.Set(echo.HeaderContentType, echo.MIMEApplicationJSON)
	rec := httptest.NewRecorder()
	ctx := e.NewContext(req, rec)

	_ = handleJugglerChecks(ctx, suite.store)
	res := ctx.Response()
	suite.Require().True(res.Status >= 200)
	suite.Require().True(res.Status < 300)
}

func (suite *HandleJugglerChecksSuite) lookup(check *juggler.HostCheck, status juggler.WalleStatus) {
	received, ok := suite.receivedChecks[check.Key()]
	suite.Assert().True(ok, "not found")
	if ok {
		suite.Assert().Equal(status, received.Status)
		suite.Assert().NotEqual(int64(0), received.EffectiveTimestamp)
	}
}

func (suite *HandleJugglerChecksSuite) TestPushSimpleStatuses() {
	suite.push(requestSimpleStatusesJSON)
	suite.lookup(&juggler.HostCheck{FQDN: "iva0-0985.search.yandex.net", Type: "bind"}, juggler.WalleCheckStatusPassed)
	suite.lookup(
		&juggler.HostCheck{FQDN: "iva0-1014.search.yandex.net", Type: "bind"},
		juggler.WalleCheckStatusSuspected,
	)
	suite.lookup(
		&juggler.HostCheck{FQDN: "man1-9419.qloud-h.yandex.net", Type: "walle_memory"},
		juggler.WalleCheckStatusFailed,
	)
}

func (suite *HandleJugglerChecksSuite) TestHealthPushFlags() {
	suite.push(requestFlagsJSON)
	suite.lookup(&juggler.HostCheck{FQDN: "iva0-0985.search.yandex.net", Type: "bind"}, juggler.WalleCheckStatusMissing)
	suite.lookup(
		&juggler.HostCheck{FQDN: "iva0-1014.search.yandex.net", Type: "bind"},
		juggler.WalleCheckStatusSuspected,
	)
	suite.lookup(
		&juggler.HostCheck{FQDN: "man1-9419.qloud-h.yandex.net", Type: "walle_memory"},
		juggler.WalleCheckStatusInvalid,
	)
}

func (suite *HandleJugglerChecksSuite) TestHealthPushInvalidMetadata() {
	suite.push(requestInvalidMetadataJSON)
	suite.lookup(
		&juggler.HostCheck{FQDN: "iva0-0985.search.yandex.net", Type: "bind"},
		juggler.WalleCheckStatusInvalid,
	)
	suite.lookup(
		&juggler.HostCheck{FQDN: "iva0-1014.search.yandex.net", Type: "bind"},
		juggler.WalleCheckStatusInvalid,
	)
}

func (suite *HandleJugglerChecksSuite) TestSaveOldStatusTime() {
	targetCheck := &juggler.HostCheck{
		FQDN:        "man2-7309.search.yandex.net",
		Type:        juggler.CheckTypeRackOverheat,
		StatusMtime: 100,
		Timestamp:   100,
		Status:      juggler.WalleCheckStatusPassed,
	}
	targetCheck.ID = targetCheck.Key()
	suite.oldChecks[targetCheck.Key()] = targetCheck
	suite.push(requestRackJSON)
	newCheckState, ok := suite.receivedChecks[targetCheck.Key()]
	suite.Require().True(ok)

	var expectedStatusTime int64 = 100
	suite.Require().Equal(expectedStatusTime, newCheckState.StatusMtime)
}

func (suite *HandleJugglerChecksSuite) TestDifferentStatusTime() {
	targetCheck := &juggler.HostCheck{
		FQDN:        "man2-7309.search.yandex.net",
		Type:        juggler.CheckTypeRackOverheat,
		StatusMtime: 100,
		Timestamp:   100,
		Status:      juggler.WalleCheckStatusFailed,
	}
	targetCheck.ID = targetCheck.Key()
	suite.oldChecks[targetCheck.Key()] = targetCheck
	suite.push(requestRackJSON)
	newCheckState, ok := suite.receivedChecks[targetCheck.Key()]
	suite.Require().True(ok)
	var oldStatusTime int64 = 100

	suite.Require().NotEqual(oldStatusTime, newCheckState.StatusMtime)
}

func (suite *HandleJugglerChecksSuite) TestExitedCheckWithCorrectJSON() {
	suite.push(failedWithCorrectJSON)
	suite.lookup(
		&juggler.HostCheck{FQDN: "wall-e.testing-man5-5A4", Type: "yt_node_gpu_check"},
		juggler.WalleCheckStatusFailed,
	)
}

func (suite *HandleJugglerChecksSuite) ibTest(reqJSON string, oracleStatus juggler.WalleStatus) *juggler.HostCheck {
	targetCheck := &juggler.HostCheck{
		FQDN:        "man2-7309.search.yandex.net",
		Type:        juggler.CheckTypeIbLink,
		StatusMtime: 100,
		Timestamp:   100,
		Status:      juggler.WalleCheckStatusFailed,
	}
	suite.push(reqJSON)

	suite.Require().Equal(1, len(suite.receivedChecks))
	suite.lookup(targetCheck, oracleStatus)
	received := suite.receivedChecks[targetCheck.Key()]
	return received
}

func (suite *HandleJugglerChecksSuite) getIbCheckMetadata(check *juggler.HostCheck) juggler.AggregatedMetadata {
	var result juggler.AggregatedMetadata
	err := json.Unmarshal([]byte(check.Metadata), &result)
	suite.Require().NoError(err)
	return result
}

func (suite *HandleJugglerChecksSuite) checkIbPorts(
	dbPorts []juggler.AggregatedMetadataCheck,
	oraclePortsStatus []juggler.JugglerStatus,
) {
	for i, port := range dbPorts {
		suite.Require().Equal(oraclePortsStatus[i], port.Status)
	}
}

func (suite *HandleJugglerChecksSuite) TestIbLinkAggregation() {
	check := suite.ibTest(requestOkIbLinkJSON, juggler.WalleCheckStatusPassed)
	metadata := suite.getIbCheckMetadata(check)
	suite.checkIbPorts(metadata.PassedChecks, []juggler.JugglerStatus{juggler.JugglerOk, juggler.JugglerOk})
	suite.Require().Equal(int64(10), check.StatusMtime) // NOTE(rocco66): oldest StatusMtime

	// NOTE(rocco66): fresh Timestamp (2000) - passive check accuracy (3 min)
	suite.Require().Equal(int64(1820), check.EffectiveTimestamp)
}

func (suite *HandleJugglerChecksSuite) TestErrIbLinkAggregation() {
	check := suite.ibTest(requestOneErrIbLinkJSON, juggler.WalleCheckStatusFailed)
	metadata := suite.getIbCheckMetadata(check)
	suite.checkIbPorts(metadata.PassedChecks, []juggler.JugglerStatus{juggler.JugglerOk})
	suite.checkIbPorts(metadata.FailedChecks, []juggler.JugglerStatus{juggler.JugglerCrit})
}

func (suite *HandleJugglerChecksSuite) prepareRackCache(rackRanges []int) {
	hostRanges := make(map[string]int)
	for i, hostRange := range rackRanges {
		hostRanges[fmt.Sprintf("host%d", i)] = hostRange
	}
	suite.rackTopologies[suite.testRackName] = hostRanges
}

func (suite *HandleJugglerChecksSuite) simpleRackRequest(critCount, okCount int) string {
	return suite.customRackRequest(
		critCount, okCount, 0, false, nil,
		juggler.CheckTypeUnreachable, juggler.OverheatMetadata{},
	)
}

func (suite *HandleJugglerChecksSuite) customRackRequest(
	critCount, okCount, warnCount int,
	insideFlappingWindow bool,
	flags []string,
	checkType juggler.CheckType,
	metadata juggler.OverheatMetadata,
) string {
	// NOTE(rocco66): CRIT first, then OK
	hostIndex := 0
	var timestamp int64
	if insideFlappingWindow {
		timestamp = time.Now().Unix() - juggler.RackCheckMapping[checkType].SuspectedPeriod - 1
	} else {
		timestamp = 0
	}
	serializedMetadata, err := json.Marshal(metadata)
	suite.Require().NoError(err)
	mkChild := func(status juggler.JugglerStatus) juggler.JugglerChild {
		hostName := fmt.Sprintf("host%d", hostIndex)
		hostIndex++
		return juggler.JugglerChild{
			HostName:     hostName,
			ServiceName:  checkType,
			InstanceName: "",
			Status:       status,
			Flags:        flags,
			Actual: juggler.JugglerChildActual{
				Status:      status,
				StatusMtime: timestamp,
				Metadata:    string(serializedMetadata),
			},
		}
	}
	var children []juggler.JugglerChild
	for i := 0; i < critCount; i++ {
		children = append(children, mkChild(juggler.JugglerCrit))
	}
	for i := 0; i < okCount; i++ {
		children = append(children, mkChild(juggler.JugglerOk))
	}
	for i := 0; i < warnCount; i++ {
		children = append(children, mkChild(juggler.JugglerWarn))
	}
	checks := [1]juggler.AggrCheckFromRequest{{
		HostName:    suite.testRackName,
		ServiceName: checkType,
		Children:    children,
	}}
	req, err := json.Marshal(juggler.HealthRequest{Checks: checks[:]})
	suite.Require().NoError(err)
	return string(req)
}

func (suite *HandleJugglerChecksSuite) getChecksSlice(checkType juggler.CheckType) []*juggler.HostCheck {
	var res []*juggler.HostCheck
	for _, check := range suite.receivedChecks {
		if check.Type == checkType {
			res = append(res, check)
		}
	}
	sort.Slice(res, func(i, j int) bool { return res[i].FQDN < res[j].FQDN })
	return res
}

func (suite *HandleJugglerChecksSuite) TestRackSplittingHalfCritHalfOk() {
	rackRanges := []int{
		1, 1, 1, 1, 1, -1, -1, -1, -1, -1,
	}
	suite.prepareRackCache(rackRanges)
	suite.push(suite.simpleRackRequest(4, 6))
	checks := suite.getChecksSlice(juggler.CheckTypeRack)
	for _, check := range checks[:5] {
		suite.Require().Equal(juggler.WalleCheckStatusFailed, check.Status)
	}
	for _, check := range checks[5:] {
		suite.Require().Equal(juggler.WalleCheckStatusPassed, check.Status)
	}
}

func (suite *HandleJugglerChecksSuite) TestRackWholeRackIsBroken() {
	rackRanges := []int{
		1, 1, 1, 1, 1, -1, -1, -1, -1, -1,
	}
	suite.prepareRackCache(rackRanges)
	suite.push(suite.simpleRackRequest(9, 1))
	checks := suite.getChecksSlice(juggler.CheckTypeRack)
	suite.Require().Equal(10, len(checks))
	for _, check := range checks {
		suite.Require().Equal(juggler.WalleCheckStatusFailed, check.Status)
	}
}

func (suite *HandleJugglerChecksSuite) flappingRackRequest(critCount, okCount int) string {
	return suite.customRackRequest(
		critCount, okCount, 0, true, nil,
		juggler.CheckTypeUnreachable, juggler.OverheatMetadata{},
	)
}

func (suite *HandleJugglerChecksSuite) TestFlappingRack() {
	suite.push(suite.flappingRackRequest(6, 4))
	checks := suite.getChecksSlice(juggler.CheckTypeRack)
	suite.Require().Equal(10, len(checks))
	for _, check := range checks {
		suite.Require().Equal(juggler.WalleCheckStatusSuspected, check.Status)
	}
}

func (suite *HandleJugglerChecksSuite) flagsRackRequest(critCount, okCount int, flags []string) string {
	return suite.customRackRequest(
		critCount, okCount, 0, false, flags,
		juggler.CheckTypeUnreachable, juggler.OverheatMetadata{},
	)
}

func (suite *HandleJugglerChecksSuite) TestMissingRack() {
	missingFlag := [...]string{juggler.CheckFilterNodata}
	suite.push(suite.flagsRackRequest(0, 10, missingFlag[:]))
	checks := suite.getChecksSlice(juggler.CheckTypeRack)
	suite.Require().Equal(10, len(checks))
	for _, check := range checks {
		suite.Require().Equal(juggler.WalleCheckStatusMissing, check.Status)
	}
}

func (suite *HandleJugglerChecksSuite) warnRackRequest(critCount, okCount, warnCount int) string {
	return suite.customRackRequest(
		critCount, okCount, warnCount, false, nil,
		juggler.CheckTypeUnreachable, juggler.OverheatMetadata{},
	)
}

func (suite *HandleJugglerChecksSuite) TestSuspectedRack() {
	suite.push(suite.warnRackRequest(0, 0, 10))
	checks := suite.getChecksSlice(juggler.CheckTypeRack)
	suite.Require().Equal(10, len(checks))
	for _, check := range checks {
		suite.Require().Equal(juggler.WalleCheckStatusSuspected, check.Status)
	}
}

func (suite *HandleJugglerChecksSuite) overheatRackRequest(critCount, okCount int) string {
	overheatMarks := [...]string{juggler.CPUOverheatingMetadataResultMark}
	return suite.customRackRequest(
		critCount, okCount, 0, false, nil,
		juggler.CheckTypeRackOverheat, juggler.OverheatMetadata{Result: overheatMarks[:]},
	)
}

func (suite *HandleJugglerChecksSuite) TestOverheatRack() {
	suite.push(suite.overheatRackRequest(10, 0))
	checks := suite.getChecksSlice(juggler.CheckTypeRackOverheat)
	suite.Require().Equal(10, len(checks))
	for _, check := range checks {
		suite.Require().Equal(juggler.WalleCheckStatusFailed, check.Status)
	}
}

func TestHandleJugglerChecks(t *testing.T) {
	suite.Run(t, new(HandleJugglerChecksSuite))
}

var requestSimpleStatusesJSON = `
{
  "checks": [
    {
      "host_name": "wall-e.testing-iva3-21",
      "service_name": "bind",
      "status": "OK",
      "hash": "224C9417B3C07831F9A68D5C5A68610F",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "iva0-0985.search.yandex.net",
          "service_name": "bind",
          "instance_name": "",
          "status": "OK",
          "description": "{\"status\": \"OK\", \"timestamp\": 1617359588.326004, \"reason\": \"\"}",
          "actual": {
            "status": "OK",
            "status_mtime": 1603993517,
            "description": "{\"status\": \"OK\", \"timestamp\": 1617359588.326004, \"reason\": \"\"}"
          },
          "flags": [
            "actual"
          ]
        },
        {
          "host_name": "iva0-1014.search.yandex.net",
          "service_name": "bind",
          "instance_name": "",
          "status": "WARN",
          "description": "{\"status\": \"WARN\", \"timestamp\": 1617359793.7417562, \"reason\": \"\"}",
          "actual": {
            "status": "WARN",
            "status_mtime": 1604074619,
            "description": "{\"status\": \"WARN\", \"timestamp\": 1617359793.7417562, \"reason\": \"\"}"
          },
          "flags": [
            "actual"
          ]
        }
      ]
    },
    {
      "host_name": "wall-e.testing-man4-4A12",
      "service_name": "walle_memory",
      "status": "OK",
      "hash": "41C1E24F7B1AC920276FBF87C70BDAE2",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "man1-9419.qloud-h.yandex.net",
          "service_name": "walle_memory",
          "instance_name": "",
          "status": "CRIT",
          "description": "{\"results\": {\"mem\": {\"status\": \"CRIT\", \"timestamp\": 1617359175.103885, \"reason\": [\"Last check: Fri Apr  2 13:26:15 2021\"]}, \"ecc\": {\"status\": \"OK\", \"timestamp\": 1617359574.538499, \"reason\": [\"Last check: Fri Apr  2 13:32:54 2021\"]}}}",
          "actual": {
            "status": "CRIT",
            "status_mtime": 1579753687,
            "description": "{\"results\": {\"mem\": {\"status\": \"CRIT\", \"timestamp\": 1617359175.103885, \"reason\": [\"Last check: Fri Apr  2 13:26:15 2021\"]}, \"ecc\": {\"status\": \"OK\", \"timestamp\": 1617359574.538499, \"reason\": [\"Last check: Fri Apr  2 13:32:54 2021\"]}}}"
          },
          "flags": [
            "actual"
          ]
        }
      ]
    }
  ]
}
`

var requestFlagsJSON = `
{
  "checks": [
    {
      "host_name": "wall-e.testing-iva3-21",
      "service_name": "bind",
      "status": "OK",
      "hash": "224C9417B3C07831F9A68D5C5A68610F",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "iva0-0985.search.yandex.net",
          "service_name": "bind",
          "instance_name": "",
          "status": "OK",
          "description": "{\"status\": \"OK\", \"timestamp\": 1617359588.326004, \"reason\": \"\"}",
          "actual": {
            "status": "OK",
            "status_mtime": 1603993517,
            "description": "{\"status\": \"OK\", \"timestamp\": 1617359588.326004, \"reason\": \"\"}"
          },
          "flags": [
            "actual", "no_data"
          ]
        },
        {
          "host_name": "iva0-1014.search.yandex.net",
          "service_name": "bind",
          "instance_name": "",
          "status": "WARN",
          "description": "{\"status\": \"WARN\", \"timestamp\": 1617359793.7417562, \"reason\": \"\"}",
          "actual": {
            "status": "WARN",
            "status_mtime": 1604074619,
            "description": "{\"status\": \"WARN\", \"timestamp\": 1617359793.7417562, \"reason\": \"\"}"
          },
          "flags": [
            "actual", "flapping"
          ]
        }
      ]
    },
    {
      "host_name": "wall-e.testing-man4-4A12",
      "service_name": "walle_memory",
      "status": "OK",
      "hash": "41C1E24F7B1AC920276FBF87C70BDAE2",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "man1-9419.qloud-h.yandex.net",
          "service_name": "walle_memory",
          "instance_name": "",
          "status": "CRIT",
          "description": "{\"results\": {\"mem\": {\"status\": \"CRIT\", \"timestamp\": 1617359175.103885, \"reason\": [\"Last check: Fri Apr  2 13:26:15 2021\"]}, \"ecc\": {\"status\": \"OK\", \"timestamp\": 1617359574.538499, \"reason\": [\"Last check: Fri Apr  2 13:32:54 2021\"]}}}",
          "actual": {
            "status": "CRIT",
            "status_mtime": 1579753687,
            "description": "{\"results\": {\"mem\": {\"status\": \"CRIT\", \"timestamp\": 1617359175.103885, \"reason\": [\"Last check: Fri Apr  2 13:26:15 2021\"]}, \"ecc\": {\"status\": \"OK\", \"timestamp\": 1617359574.538499, \"reason\": [\"Last check: Fri Apr  2 13:32:54 2021\"]}}}"
          },
          "flags": [
            "actual", "invalid"
          ]
        }
      ]
    }
  ]
}
`
var requestInvalidMetadataJSON = `
{
  "checks": [
    {
      "host_name": "wall-e.testing-iva3-21",
      "service_name": "bind",
      "status": "OK",
      "hash": "224C9417B3C07831F9A68D5C5A68610F",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "iva0-0985.search.yandex.net",
          "service_name": "bind",
          "instance_name": "",
          "status": "OK",
          "description": "{\"status\": \"OK\", \"timestamp\": 1617359588.326004, \"reason\": \"\"}",
          "actual": {
            "status": "OK",
            "status_mtime": 1603993517,
            "description": "timed out after some"
          },
          "flags": [
            "actual", "no_data"
          ]
        },
        {
          "host_name": "iva0-1014.search.yandex.net",
          "service_name": "bind",
          "instance_name": "",
          "status": "WARN",
          "description": "{\"status\": \"WARN\", \"timestamp\": 1617359793.7417562, \"reason\": \"\"}",
          "actual": {
            "status": "WARN",
            "status_mtime": 1604074619,
            "description": "exited with code some"
          },
          "flags": [
            "actual", "flapping"
          ]
        }
      ]
    }
  ]
}
`
var requestRackJSON = `
{
  "checks": [
    {
      "host_name": "wall-e.testing-man5-5A4",
      "service_name": "walle_cpu_caches",
      "status": "OK",
      "hash": "31AFC986FF69017B672C47E286F92FF8",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "man2-7309.search.yandex.net",
          "service_name": "walle_cpu_caches",
          "instance_name": "",
          "status": "OK",
          "description": "{\"result\": {\"status\": \"OK\", \"raw\": \"\", \"reason\": [\"Last check: Fri Apr  2 13:31:43 2021\"], \"socket\": null, \"timestamp\": 1617359503.470452}}",
          "actual": {
            "status": "OK",
            "status_mtime": 1579753634,
            "description": "{\"result\": {\"status\": \"OK\", \"raw\": \"\", \"reason\": [\"Last check: Fri Apr  2 13:31:43 2021\"], \"socket\": null, \"timestamp\": 1617359503.470452}}"
          },
          "flags": [
            "actual"
          ]
        }
      ]
    }
  ]
}
`

var failedWithCorrectJSON = `
{
  "checks": [
    {
      "host_name": "wall-e.testing-man5-5A4",
      "service_name": "yt_node_gpu_check",
      "status": "OK",
      "hash": "31AFC986FF69017B672C47E286F92FF8",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "wall-e.testing-man5-5A4",
          "service_name": "yt_node_gpu_check",
          "instance_name": "",
          "status": "CRIT",
          "description": "{\"reason\": \"preliminary GPU check command failed: process exited with code 1\", \"timestamp\": 1617359503.470452}",
          "actual": {
            "status": "CRIT",
            "status_mtime": 1579753634,
			"description": "{\"reason\": \"preliminary GPU check command failed: process exited with code 1\", \"timestamp\": 1617359503.470452}"
          },
          "flags": [
            "actual"
          ]
        }
      ]
    }
  ]
}
`

var requestOneErrIbLinkJSON = `
{
  "checks": [
    {
      "host_name": "wall-e.testing-man5-5A4",
      "service_name": "walle_ib_link",
      "status": "OK",
      "hash": "31AFC986FF69017B672C47E286F92FF8",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "man2-7309.search.yandex.net",
          "service_name": "walle_ib_link",
          "instance_name": "port1",
          "status": "CRIT",
          "description": "",
          "actual": {
            "status": "CRIT",
            "status_mtime": 1579753634,
            "description": ""
          },
          "flags": [
            "actual"
          ]
        },
        {
          "host_name": "man2-7309.search.yandex.net",
          "service_name": "walle_ib_link",
          "instance_name": "port2",
          "status": "OK",
          "description": "",
          "actual": {
            "status": "OK",
            "status_mtime": 1579753634,
            "description": ""
          },
          "flags": [
            "actual"
          ]
        }
      ]
    }
  ]
}
`

var requestOkIbLinkJSON = `
{
  "checks": [
    {
      "host_name": "wall-e.testing-man5-5A4",
      "service_name": "walle_ib_link",
      "status": "OK",
      "hash": "31AFC986FF69017B672C47E286F92FF8",
      "description": "",
      "flags": [
        "actual"
      ],
      "children": [
        {
          "host_name": "man2-7309.search.yandex.net",
          "service_name": "walle_ib_link",
          "instance_name": "port1",
          "status": "OK",
          "description": "",
          "actual": {
            "status": "OK",
            "status_mtime": 10,
            "description": "{\"status\":\"OK\",\"timestamp\":2000,\"reason\":\"No errs\"}"
          },
          "flags": [
            "actual"
          ]
        },
        {
          "host_name": "man2-7309.search.yandex.net",
          "service_name": "walle_ib_link",
          "instance_name": "port2",
          "status": "OK",
          "description": "",
          "actual": {
            "status": "OK",
            "status_mtime": 20,
            "description": "{\"status\":\"OK\",\"timestamp\":1000,\"reason\":\"No errs\"}"
          },
          "flags": [
            "actual"
          ]
        }
      ]
    }
  ]
}
`
