package api

import (
	"context"
	"sync"
	"time"

	"go.mongodb.org/mongo-driver/mongo"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	"a.yandex-team.ru/library/go/core/log"
)

const jugglerChecksQueueSize = 10000

type HealthRepo interface {
	NewBulkWriter(options *repos.HealthBulkWriterOptions) (repos.HealthBulkWriter, error)
	FindIDMap(ctx context.Context, filter *repos.HostCheckFilter) (map[juggler.WalleCheckKey]*juggler.HostCheck, error)
}

type CacheRepo interface {
	FindRackTopologies(ctx context.Context) (map[string]*repos.RackTopology, error)
}

type store struct {
	healthRepo     HealthRepo
	writer         repos.HealthBulkWriter
	rackTopologies *rackTopologyStore
	queue          chan []*juggler.HostCheck
	logger         log.Logger
}

func newStore(base *lib.Application, cacheRepo CacheRepo, healthRepo HealthRepo) (*store, error) {
	writer, err := defaultWriter(base, healthRepo)
	if err != nil {
		return nil, err
	}
	return &store{
		healthRepo: healthRepo,
		writer:     writer,
		rackTopologies: &rackTopologyStore{
			cacheRepo:  cacheRepo,
			localCache: make(map[string]*repos.RackTopology),
			logger:     base.Logger("base"),
		},
		queue:  make(chan []*juggler.HostCheck, jugglerChecksQueueSize),
		logger: base.Logger("base"),
	}, nil
}

func (s *store) FindIDMap(ctx context.Context, filter *repos.HostCheckFilter) (
	map[juggler.WalleCheckKey]*juggler.HostCheck,
	error,
) {
	return s.healthRepo.FindIDMap(ctx, filter)
}

func (s *store) NewBulkWriter(options *repos.HealthBulkWriterOptions) (repos.HealthBulkWriter, error) {
	return s.healthRepo.NewBulkWriter(options)
}

func (s *store) QueueIsFull() bool {
	return cap(s.queue) == len(s.queue)
}

func (s *store) TryPush(checks []*juggler.HostCheck) bool {
	select {
	case s.queue <- checks:
		return true
	default:
		return false
	}
}

func (s *store) RackTopologyStore() juggler.RackTopologyStore {
	return s.rackTopologies
}

func (s *store) updateRackTopologies(ctx context.Context) {
	ticker := time.NewTicker(10 * time.Second)
	defer ticker.Stop()
	for {
		if err := s.rackTopologies.refresh(ctx); err != nil {
			s.logger.Errorf("refresh rack topologies: %v", err)
		}
		select {
		case <-ctx.Done():
			return
		case <-ticker.C:
		}
	}
}

func (s *store) write(ctx context.Context) {
	go s.writer.Run()
	metricTicker := time.NewTicker(time.Second)
	defer metricTicker.Stop()
	for {
		select {
		case <-metricTicker.C:
			metrics.FreeAggrBuffer.Update(float64(cap(s.queue) - len(s.queue)))
		case checks := <-s.queue:
			s.writer.Add(checks)
		case <-ctx.Done():
			s.writer.Shutdown()
			return
		}
	}
}

func defaultWriter(base *lib.Application, repo HealthRepo) (repos.HealthBulkWriter, error) {
	return repo.NewBulkWriter(&repos.HealthBulkWriterOptions{
		Size:          4000,
		FlushInterval: 2 * time.Second,
		RPSLimit:      3,
		Handler: func(result *mongo.BulkWriteResult, err error) {
			if err != nil {
				base.Logger("base").Errorf("write host checks to DB: %v", err)
			}
			if result != nil {
				base.Logger("base").Infof("host checks was written to DB (%d inserted, %d modified)",
					result.UpsertedCount,
					result.ModifiedCount)
			}
		},
	})
}

type rackTopologyStore struct {
	cacheRepo  CacheRepo
	logger     log.Logger
	mu         sync.RWMutex
	localCache map[string]*repos.RackTopology
}

func (r *rackTopologyStore) refresh(ctx context.Context) error {
	tmp, err := r.cacheRepo.FindRackTopologies(ctx)
	if err != nil {
		return err
	}
	r.logger.Infof("rack local cache size: %d", len(tmp))
	r.mu.Lock()
	r.localCache = tmp
	r.mu.Unlock()
	return nil
}

func (r *rackTopologyStore) Get(rackName string) map[string]int {
	r.mu.RLock()
	defer r.mu.RUnlock()
	if r.localCache[rackName] != nil {
		return r.localCache[rackName].HostRanges
	}
	r.logger.Warnf("rack %s not found", rackName)
	return nil
}
