package api

import (
	"context"
	"errors"
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/httputil"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
)

const (
	dataExpiration      = 5 * time.Minute
	metricGroupEndpoint = "metric_group"
)

var formatFields = map[string]string{
	"yasm":    repos.MetricsFieldKeyYasmData,
	"solomon": repos.MetricsFieldKeySolomonData,
}

type MetricRepo interface {
	GetData(ctx context.Context, filter *repos.MetricFilter, dataField string) ([]byte, error)
}

func handle(ctx echo.Context, repo MetricRepo) error {
	groupID := ctx.Param(metricGroupEndpoint)
	format := ctx.QueryParam("format")
	if err := validate(groupID, format); err != nil {
		return httputil.RespondError(ctx, http.StatusBadRequest, err, "validate")
	}
	data, err := repo.GetData(
		ctx.Request().Context(),
		&repos.MetricFilter{ID: groupID, TimeAfter: time.Now().Add(-dataExpiration).Unix()},
		formatFields[format],
	)
	if err != nil {
		return httputil.RespondError(ctx, http.StatusInternalServerError, err, "load data")
	}
	return ctx.Blob(http.StatusOK, echo.MIMEApplicationJSON, data)
}

func validate(groupID, format string) error {
	switch {
	case groupID == "":
		return errors.New("no metric group")
	case formatFields[format] == "":
		return errors.New("unknown format")
	default:
		return nil
	}
}
