package cron

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"time"

	"github.com/labstack/echo/v4"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
	libcron "a.yandex-team.ru/infra/walle/server/go/internal/lib/cron"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/httputil"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
	"a.yandex-team.ru/infra/walle/server/go/internal/metrics"
	statjob "a.yandex-team.ru/infra/walle/server/go/internal/statistics/job"
)

const (
	HostStateEventSnapshotJob = "host_state_event_snapshot"
	HostProjectSnapshotJob    = "host_project_snapshot"
	ProjectTagSnapshotJob     = "project_tag_snapshot"
	CollectMetricsJob         = "collect_metrics"
)

type application struct {
	base                *lib.Application
	config              *libcron.Config
	server              *http.Server
	mongoDB             *mongo.Database
	mongoDBHealth       *mongo.Database
	mongoReadPref       *readpref.ReadPref
	mongoReadPrefHealth *readpref.ReadPref
	ydb                 *db.YDBClient
	jugglerAgentClient  monitoring.JugglerAgentClient
	registry            *registry
}

func StartApplication(args *lib.CliArgs) {
	if err := os.Chdir(args.RootDir); err != nil {
		panic(err)
	}
	config := &libcron.Config{}
	if err := lib.LoadConfig(args.ConfigPath, config); err != nil {
		panic(err)
	}
	app, err := initApplication(config)
	if err != nil {
		panic(fmt.Errorf("init app: %v", err))
	}
	if err := app.run(); err != nil {
		app.base.Logger("base").Errorf("fatal: %v", err)
		panic(err)
	}
}

func initApplication(config *libcron.Config) (*application, error) {
	app := &application{
		config: config,
	}
	baseApp, err := lib.NewApplication(config.BaseApp)
	if err != nil {
		return nil, err
	}
	app.base = baseApp

	e := echo.New()
	e.Use(func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			defer func() {
				if err := recover(); err != nil {
					baseApp.Logger("base").Errorf("recover from panic: %v", err)
				}
			}()
			return next(ctx)
		}
	})
	e.GET("/ping", monitoring.NewHealthCheckHTTPHandler())
	e.GET("/unistat", monitoring.NewUnistatHTTPHandler())
	if app.server, err = httputil.NewServer(config.HTTPServer, e); err != nil {
		return nil, err
	}
	if app.jugglerAgentClient, err = monitoring.NewJugglerAgentClient(config.JugglerAgent); err != nil {
		return nil, err
	}

	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	if app.mongoDB, err = db.NewMongoConnect(ctx, config.Mongo); err != nil {
		return nil, fmt.Errorf("connect to mongo: %v", err)
	}
	if app.mongoDBHealth, err = db.NewMongoConnect(ctx, config.MongoHealth); err != nil {
		return nil, fmt.Errorf("connect to mongo: %v", err)
	}
	app.mongoReadPref = db.NewMongoLocalReadPreference(config.Mongo.GeoTags)
	app.mongoReadPrefHealth = db.NewMongoLocalReadPreference(config.MongoHealth.GeoTags)

	if app.ydb, err = db.NewYDBClient(ctx, config.YDB); err != nil {
		return nil, fmt.Errorf("connect to YDB: %v", err)
	}
	if err = app.initJobRegistry(); err != nil {
		return nil, err
	}
	return app, nil
}

func (app *application) initJobRegistry() error {
	app.registry = newRegistry(app.base.Logger("base"), app.jugglerAgentClient)
	store := libcron.NewStore(app.ydb, app.mongoDB, app.mongoDBHealth, app.mongoReadPref, app.mongoReadPrefHealth)
	jobs := map[string]libcron.NewJob{
		ProjectTagSnapshotJob:     statjob.NewProjectTagSnapshotJob,
		HostProjectSnapshotJob:    statjob.NewHostProjectSnapshotJob,
		HostStateEventSnapshotJob: statjob.NewHostEventSnapshotJob,
		CollectMetricsJob:         metrics.NewCollectMetricsJob,
	}
	for name, conf := range app.config.Registry.Jobs {
		job, err := jobs[name](&libcron.JobConfig{
			Store:        store,
			Logger:       app.base.Logger("base"),
			Extra:        conf.Extra,
			GlobalConfig: app.config,
		})
		if err != nil {
			return err
		}
		app.registry.jobs[name] = job
	}
	return app.registry.configure(app.config.Registry)
}

func (app *application) run() error {
	defer app.close()
	app.base.AddServeFunc("registry", app.registry.run)
	app.base.AddServeFunc("http", httputil.NewServeFunc(app.config.HTTPServer, app.server))
	return app.base.Run()
}

func (app *application) close() {
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	_ = app.mongoDB.Client().Disconnect(ctx)
	_ = app.mongoDBHealth.Client().Disconnect(ctx)
	_ = app.ydb.Close()
	app.jugglerAgentClient.Stop()
}
