package cron

import (
	"context"
	"fmt"
	"time"

	libcron "a.yandex-team.ru/infra/walle/server/go/internal/lib/cron"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
	"a.yandex-team.ru/library/go/core/log"
)

type registry struct {
	scheduler  *libcron.Scheduler
	jobs       map[string]libcron.Job
	monitoring *libcron.Monitoring
	logger     log.Logger
}

func newRegistry(logger log.Logger, jugglerAgent monitoring.JugglerAgentClient) *registry {
	return &registry{
		scheduler:  libcron.NewScheduler(logger),
		jobs:       make(map[string]libcron.Job),
		monitoring: libcron.NewMonitoring(jugglerAgent),
		logger:     logger,
	}
}

func (r *registry) configure(config libcron.RegistryConfig) error {
	for name, conf := range config.Jobs {
		decorators := []libcron.DecoratorFunc{
			func(next libcron.JobFunc) libcron.JobFunc {
				name := name
				return func(ctx context.Context) (bool, error) {
					r.logger.Infof("%s started", name)
					executed, err := next(ctx)
					r.logger.Infof("%s ended: execution: %v", name, executed)
					return executed, err
				}
			},
		}

		if conf.Monitoring != nil {
			critTimeout, err := time.ParseDuration(conf.Monitoring.CritTimeout)
			if err != nil {
				return fmt.Errorf("parse critical timeout: %w", err)
			}
			decorators = append(decorators, r.monitoring.NewDecorator(name, critTimeout))
		}
		metrics := newMetricsGroup(name)
		if err := monitoring.RegisterMetrics(metrics); err != nil {
			return fmt.Errorf("register metrics: %w", err)
		}
		decorators = append(decorators, executionTimeDecorator(metrics.ExecutionTime))

		if err := r.scheduler.Add(name, conf.Spec, r.jobs[name].Do, decorators...); err != nil {
			return fmt.Errorf("configure %s job: %w", name, err)
		}
	}
	return nil
}

func (r *registry) run(ctx context.Context) {
	r.scheduler.Start()
	<-ctx.Done()
	r.scheduler.Stop()
}
