package cron

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/dmc/screening"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/httputil"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	lockrepo "a.yandex-team.ru/infra/walle/server/go/internal/utilities/repository"
)

type config struct {
	BaseApp      lib.AppConfig                 `yaml:",inline"`
	HTTPServer   httputil.ServerConfig         `yaml:"http_server"`
	Mongo        db.MongoConfig                `yaml:"mongo"`
	MongoHealth  db.MongoConfig                `yaml:"mongo_health"`
	YDB          db.YDBConfig                  `yaml:"ydb"`
	JugglerAgent monitoring.JugglerAgentConfig `yaml:"juggler_agent"`
	ExpertSystem *lib.ExpertSystemConfig       `yaml:"expert_system"`
}

type application struct {
	base                *lib.Application
	screening           *screening.Screening
	config              *config
	server              *http.Server
	mongoDB             *mongo.Database
	mongoDBHealth       *mongo.Database
	mongoReadPref       *readpref.ReadPref
	mongoReadPrefHealth *readpref.ReadPref
	ydb                 *db.YDBClient
	jugglerAgentClient  monitoring.JugglerAgentClient
}

func StartApplication(args *lib.CliArgs) {
	if err := os.Chdir(args.RootDir); err != nil {
		panic(err)
	}
	config := &config{}
	if err := lib.LoadConfig(args.ConfigPath, config); err != nil {
		panic(err)
	}
	app, err := initApplication(config)
	if err != nil {
		panic(fmt.Errorf("init app: %v", err))
	}
	if err := app.run(); err != nil {
		app.base.Logger("base").Errorf("fatal: %v", err)
		panic(err)
	}
}

func initApplication(config *config) (*application, error) {
	app := &application{
		config: config,
	}
	baseApp, err := lib.NewApplication(config.BaseApp)
	if err != nil {
		return nil, err
	}
	app.base = baseApp

	e := echo.New()
	e.Use(func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			defer func() {
				if err := recover(); err != nil {
					baseApp.Logger("base").Errorf("recover from panic: %v", err)
				}
			}()
			return next(ctx)
		}
	})
	e.GET("/ping", monitoring.NewHealthCheckHTTPHandler())
	e.GET("/unistat", monitoring.NewUnistatHTTPHandler())
	if app.server, err = httputil.NewServer(config.HTTPServer, e); err != nil {
		return nil, err
	}
	if app.jugglerAgentClient, err = monitoring.NewJugglerAgentClient(config.JugglerAgent); err != nil {
		return nil, err
	}

	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	if app.mongoDB, err = db.NewMongoConnect(ctx, config.Mongo); err != nil {
		return nil, fmt.Errorf("connect to mongo: %v", err)
	}
	if app.mongoDBHealth, err = db.NewMongoConnect(ctx, config.MongoHealth); err != nil {
		return nil, fmt.Errorf("connect to mongo: %v", err)
	}
	app.mongoReadPref = db.NewMongoLocalReadPreference(config.Mongo.GeoTags)
	app.mongoReadPrefHealth = db.NewMongoLocalReadPreference(config.MongoHealth.GeoTags)

	if app.ydb, err = db.NewYDBClient(ctx, config.YDB); err != nil {
		return nil, fmt.Errorf("connect to YDB: %v", err)
	}

	tier, err := strconv.ParseUint(app.config.BaseApp.Tier, 10, 64)
	if err != nil {
		return nil, fmt.Errorf("can't parse tier: %v", err)
	}
	shardsNum, ok := config.ExpertSystem.ShardsNum[int(tier)]
	if !ok {
		shardsNum = config.ExpertSystem.ShardsNumDefault
	}
	app.screening, err = screening.NewScreening(
		app.base.Logger("base"),
		shardsNum,
		repos.NewHostRepo(app.mongoDB, app.mongoReadPref),
		repos.NewHostDecisionRepo(app.mongoDB, app.mongoReadPref),
		repos.NewHealthRepo(app.mongoDBHealth, app.mongoReadPrefHealth),
		repos.NewOperationLogRepo(app.mongoDB, app.mongoReadPref),
		lockrepo.NewLockRepository(app.ydb),
		tier,
	)
	if err != nil {
		return nil, err
	}
	return app, nil
}

func (app *application) run() error {
	defer app.close()
	app.base.AddServeFunc("screening", app.screening.Run)
	app.base.AddServeFunc("http", httputil.NewServeFunc(app.config.HTTPServer, app.server))
	return app.base.Run()
}

func (app *application) close() {
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	_ = app.mongoDB.Client().Disconnect(ctx)
	_ = app.mongoDBHealth.Client().Disconnect(ctx)
	_ = app.ydb.Close()
	app.jugglerAgentClient.Stop()
}
