package screening

import (
	"context"
	"fmt"
	"sync"
	"time"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/collections"
	"a.yandex-team.ru/infra/walle/server/go/internal/utilities"
	lockrepo "a.yandex-team.ru/infra/walle/server/go/internal/utilities/repository"

	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	"a.yandex-team.ru/library/go/core/log"
)

const mainIterationInterval = 2 * time.Minute

type Screening struct {
	logger           log.Logger
	hostsRepo        *repos.HostRepo
	hostDecisionRepo *repos.HostDecisionRepo
	healthRepo       *repos.HealthRepo
	operationLogRepo *repos.OperationLogRepo
	locker           utilities.Locker
	tier             uint64
	shardManager     *shardManager
}

func NewScreening(
	logger log.Logger,
	shardsNum int,
	hostRepo *repos.HostRepo,
	hostDecisionRepo *repos.HostDecisionRepo,
	healthRepo *repos.HealthRepo,
	operationLogRepo *repos.OperationLogRepo,
	lockRepo lockrepo.LockRepo,
	tier uint64,
) (*Screening, error) {
	manager, err := newShardManager(tier, shardsNum, lockRepo, logger)
	if err != nil {
		return nil, err
	}
	return &Screening{
		logger:           logger,
		hostsRepo:        hostRepo,
		hostDecisionRepo: hostDecisionRepo,
		healthRepo:       healthRepo,
		operationLogRepo: operationLogRepo,
		locker:           utilities.NewLocker(lockRepo),
		tier:             tier,
		shardManager:     manager,
	}, nil
}

func (screening *Screening) Run(ctx context.Context) {
	defer screening.shardManager.clean()
	ticker := time.NewTicker(mainIterationInterval)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			screening.logger.Info("DMC screening iteration start")
			shards, err := screening.shardManager.getShards(ctx)
			if err != nil {
				screening.logger.Errorf("Can't get shards: %v", err)
				continue
			}
			wg := &sync.WaitGroup{}
			for _, item := range shards {
				wg.Add(1)
				go func(shard *shard) {
					defer wg.Done()
					if err := screening.processShard(ctx, shard); err != nil {
						screening.logger.Errorf("Fail to process shard %d: %v", shard.id, err)
					}
				}(item)
			}
			wg.Wait()
		case <-ctx.Done():
			return
		}
	}
}

func (screening *Screening) processShard(ctx context.Context, shard *shard) error {
	lock, err := screening.locker.Lock(
		screening.logger,
		fmt.Sprintf("dmc-screening-%d/shard-%d", screening.tier, shard.id),
		screening.shardManager.instance,
	)
	if err != nil {
		return err
	}
	defer screening.locker.Unlock(lock)

	// TODO(rocco66): all screening filters from python screening
	filter := repos.HostFilter{
		Tier:  &repos.HostFilterTier{TierValue: screening.tier},
		Shard: &repos.HostFilterShard{ID: shard.id, Num: screening.shardManager.shardsNum},
	}
	hosts, err := screening.hostsRepo.Find(ctx, &filter)
	if err != nil {
		return err
	}
	return screening.processHosts(ctx, hosts)
}

func (screening *Screening) processHosts(ctx context.Context, hosts []*repos.Host) error {
	var rulesCache = make(RulesCache)
	for _, hostsChunk := range collections.Chunks(hosts, 200) {
		reqs := WorldRequirements{}
		for _, host := range hostsChunk {
			rules := GetProjectRules(host.Project, rulesCache)
			for _, rule := range rules {
				reqs.Add(host, rule.GetDeps())
			}
		}
		world, err := reqs.GetWorldState(ctx, screening.healthRepo, screening.operationLogRepo)
		if err != nil {
			return fmt.Errorf("can't get world state for some hosts (e.g. %v)", hostsChunk[:5])
		}
		for _, host := range hostsChunk {
			rules := GetProjectRules(host.Project, rulesCache)
			decision := HandleHost(host, world, rules)
			// TODO bulk
			err := screening.hostDecisionRepo.Upsert(ctx, host.UUID, decision)
			if err != nil {
				screening.logger.Errorf("can't write decision %v %s", host, err)
			}
		}
	}
	return nil
}
