package lib

import (
	"context"
	"fmt"
	"io"
	"os"
	"os/signal"
	"syscall"

	"go.uber.org/zap/zapio"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

type CliArgs struct {
	RootDir    string
	ConfigPath string
	DryRun     bool
}

type AppConfig struct {
	Env     string                  `yaml:"env"`
	Debug   bool                    `yaml:"debug"`
	Loggers map[string]LoggerConfig `yaml:"loggers"`
	Tier    string                  `yaml:"tier"`
}

type ExpertSystemConfig struct {
	ShardsNum        map[int]int `yaml:"shards_num"`
	ShardsNumDefault int         `yaml:"shards_num_default"`
}

type ServeFunc func(context.Context)

type Application struct {
	Env        string
	Debug      bool
	Loggers    map[string]*zap.Logger
	serveFuncs map[string]ServeFunc
}

func NewApplication(config AppConfig) (*Application, error) {
	loggers, err := NewLoggers(config.Loggers)
	if err != nil {
		return nil, err
	}

	app := &Application{
		Env:        config.Env,
		Debug:      config.Debug,
		Loggers:    loggers,
		serveFuncs: make(map[string]ServeFunc),
	}

	return app, nil
}

func (app *Application) Logger(name string) log.Logger {
	return app.Loggers[name]
}

func (app *Application) LoggerWriter(name string) io.Writer {
	return &zapio.Writer{Log: app.Loggers[name].L}
}

func (app *Application) AddServeFunc(name string, f ServeFunc) {
	app.serveFuncs[name] = f
}

func (app *Application) Run() error {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	g, ctx := errgroup.WithContext(ctx)
	for name, fn := range app.serveFuncs {
		name := name
		f := fn
		g.Go(func() error {
			var err error
			func() {
				defer func() {
					if r := recover(); r != nil {
						err = fmt.Errorf("serving %s down: %v", name, r)
					}
				}()
				f(ctx)
			}()
			return err
		})
	}

	quit := make(chan os.Signal, 1)
	signal.Notify(quit, os.Interrupt, syscall.SIGTERM)
	select {
	case <-quit:
		cancel()
	case <-ctx.Done():
	}
	return g.Wait()
}
