package automation

import (
	"strings"

	"a.yandex-team.ru/infra/walle/server/go/internal/repos"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler"
)

type HostState string

const (
	HostStateAssigned    HostState = "assigned"
	HostStateMaintenance HostState = "maintenance"
	HostStateProbation   HostState = "probation"
)

type HostStatus string

const (
	HostStatusDead    HostStatus = "dead"
	HostStatusInvalid HostStatus = "invalid"
)

type HostHealthStatus string

const (
	HostHealthStatusOk HostHealthStatus = "ok"
)

var assignedStates = map[HostState]struct{}{
	HostStateAssigned:    {},
	HostStateMaintenance: {},
	HostStateProbation:   {},
}

type TaskType string

const (
	TaskTypeManual TaskType = "manual"
)

type Host struct {
	ProjectID    repos.ProjectID
	Tier         int
	State        HostState
	Status       HostStatus
	Restrictions []HostRestriction
	Health       *HostHealth
	Inv          int
	Task         *HostTask
}

type HostHealth struct {
	Status   HostHealthStatus
	Decision *HostHealthDecision
	Reasons  []string
}

type HostHealthDecision struct {
	Action string
}

type HostTask struct {
	Type TaskType
}

func (h *Host) IsAutomated() bool {
	if !(h.State == HostStateAssigned || h.State == HostStateProbation) {
		return false
	}
	if h.Status == HostStatusInvalid {
		return false
	}
	for _, r := range h.Restrictions {
		if r, ok := restrictions[r]; ok && r.automated {
			return false
		}
	}
	return true
}

func (h *Host) IsHealthy() bool {
	if _, ok := assignedStates[h.State]; !ok {
		return false
	}
	if h.Health == nil {
		return false
	}
	if h.Health.Status != HostHealthStatusOk {
		return false
	}
	return true
}

func (h *Host) IsHealthMissing() bool {
	if _, ok := assignedStates[h.State]; !ok {
		return false
	}
	if h.Status == HostStatusInvalid {
		return false
	}
	if h.Health != nil {
		return false
	}
	return true
}

func GetShardID(inv int, shardsNum int) int {
	return inv % shardsNum
}

func SplitFailureReason(reason string) (string, juggler.WalleStatus) {
	sepIndex := strings.LastIndex(reason, ".")
	if sepIndex != -1 {
		return reason[:sepIndex], juggler.WalleStatus(reason[sepIndex+1:])
	}
	if check, status := splitFailureReasonDepricated(reason); check != "" && status != "" {
		return check, status
	}
	return "", ""
}

func splitFailureReasonDepricated(reason string) (string, juggler.WalleStatus) {
	sepIndex := strings.LastIndex(reason, "-")
	if sepIndex != -1 {
		return reason[:sepIndex], juggler.WalleStatus(reason[sepIndex+1:])
	}
	return "", ""
}
