package cron

import (
	"context"
	"fmt"
	"time"

	"go.uber.org/atomic"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
)

const (
	monitoringTimeTemplate = "02.01.2006 15:04:05"
)

type Monitoring struct {
	jugglerAgentClient monitoring.JugglerAgentClient
	stat               map[string]*cronJobMonitoringStat
	config             map[string]*cronJobMonitoringConfig
}

type cronJobMonitoringConfig struct {
	CritTimeout time.Duration
}

type cronJobMonitoringStat struct {
	FailedRun  *atomic.Time
	SuccessRun *atomic.Time
}

func NewMonitoring(jugglerAgentClient monitoring.JugglerAgentClient) *Monitoring {
	return &Monitoring{
		jugglerAgentClient: jugglerAgentClient,
		stat:               make(map[string]*cronJobMonitoringStat),
		config:             make(map[string]*cronJobMonitoringConfig),
	}
}

func (m *Monitoring) NewDecorator(id string, critTimeout time.Duration) DecoratorFunc {
	if _, ok := m.config[id]; !ok {
		m.stat[id] = &cronJobMonitoringStat{
			SuccessRun: atomic.NewTime(time.Now()),
			FailedRun:  atomic.NewTime(time.Time{}),
		}
		m.config[id] = &cronJobMonitoringConfig{CritTimeout: critTimeout}
	}

	return func(next JobFunc) JobFunc {
		return func(ctx context.Context) (bool, error) {
			defer func() {
				if err := recover(); err != nil {
					m.handleFail(id)
					panic(err)
				}
			}()

			startedAt := time.Now()
			executed, err := next(ctx)
			elapsed := time.Since(startedAt)
			if executed {
				m.do(id, err != nil, elapsed)
			}
			return executed, err
		}
	}
}

func (m *Monitoring) do(job string, fail bool, duration time.Duration) {
	switch {
	case fail:
		m.handleFail(job)
	case duration > m.config[job].CritTimeout:
		m.handleLongExecution(job, duration)
	default:
		m.handleOk(job)

	}
}

func (m *Monitoring) handleFail(job string) {
	successRun := m.stat[job].SuccessRun.Load()
	failedRun := time.Now()
	m.stat[job].FailedRun.Store(failedRun)
	info := fmt.Sprintf(
		"Cron job %s fail: last success run: %s; last failure run: %s.",
		job,
		successRun.Format(monitoringTimeTemplate),
		failedRun.Format(monitoringTimeTemplate),
	)
	_ = m.jugglerAgentClient.Push(&monitoring.JugglerEvent{
		Service:     MonitoringServiceSuffix(job),
		Status:      monitoring.JugglerStatusCrit,
		Description: info,
	})
}

func (m *Monitoring) handleOk(job string) {
	successRun := time.Now()
	failedRun := m.stat[job].FailedRun.Load()
	m.stat[job].SuccessRun.Store(successRun)
	info := fmt.Sprintf(
		"Cron job %s runs successfully: last success run: %s; last failure run: %s",
		job,
		successRun.Format(monitoringTimeTemplate),
		failedRun.Format(monitoringTimeTemplate),
	)
	_ = m.jugglerAgentClient.Push(&monitoring.JugglerEvent{
		Service:     MonitoringServiceSuffix(job),
		Status:      monitoring.JugglerStatusOk,
		Description: info,
	})
}

func (m *Monitoring) handleLongExecution(job string, duration time.Duration) {
	info := fmt.Sprintf(
		"Cron job %s run to long: %v",
		job,
		duration,
	)
	_ = m.jugglerAgentClient.Push(&monitoring.JugglerEvent{
		Service:     MonitoringServiceSuffix(job),
		Status:      monitoring.JugglerStatusCrit,
		Description: info,
	})
}

func MonitoringServiceSuffix(jobName string) string {
	return fmt.Sprintf("cron.%s.last_success_run", jobName)
}
