package cron

import (
	"context"
	"fmt"
	"testing"
	"time"

	"github.com/stretchr/testify/suite"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
)

type MonitoringSuite struct {
	suite.Suite
}

type mockJugglerClient struct {
	results map[string]monitoring.JugglerStatus
}

func (client *mockJugglerClient) Push(event *monitoring.JugglerEvent) error {
	client.results[event.Service] = event.Status
	return nil
}

func (client *mockJugglerClient) Stop() {}

func (suite *MonitoringSuite) TestMonitoring() {
	client := &mockJugglerClient{results: make(map[string]monitoring.JugglerStatus)}
	m := NewMonitoring(client)
	id1 := "service-1"
	id2 := "service-2"
	d1 := m.NewDecorator(id1, time.Hour)
	d2 := m.NewDecorator(id2, 0)
	jobFuncs := map[string]JobFunc{
		"normal": func(context.Context) (bool, error) { return true, nil },
		"error":  func(context.Context) (bool, error) { return true, fmt.Errorf("error") },
	}
	type results = map[string]monitoring.JugglerStatus
	type testcase struct {
		jobFunc JobFunc
		results results
	}
	id1 = MonitoringServiceSuffix(id1)
	id2 = MonitoringServiceSuffix(id2)
	cases := []testcase{
		{
			jobFunc: d1(jobFuncs["normal"]),
			results: results{
				id1: monitoring.JugglerStatusOk,
			},
		},
		{
			jobFunc: d1(jobFuncs["error"]),
			results: results{
				id1: monitoring.JugglerStatusCrit,
			},
		},
		{
			jobFunc: d2(jobFuncs["normal"]),
			results: results{
				id2: monitoring.JugglerStatusCrit,
			},
		},
		{
			jobFunc: d2(jobFuncs["error"]),
			results: results{
				id2: monitoring.JugglerStatusCrit,
			},
		},
	}
	ctx := context.Background()
	for _, c := range cases {
		_, _ = c.jobFunc(ctx)
		suite.Equal(c.results, client.results)
		client.results = make(map[string]monitoring.JugglerStatus)
	}
}

func TestMonitoring(t *testing.T) {
	suite.Run(t, new(MonitoringSuite))
}
