package cron

import (
	"fmt"
	"strconv"
	"strings"
	"time"
)

type Spec struct {
	Every string `yaml:"every"`
	From  string `yaml:"from"`
}

type schedule struct {
	duration time.Duration
	second   int
	minute   int
	hour     int
}

func parse(spec *Spec) (*schedule, error) {
	duration, err := time.ParseDuration(spec.Every)
	if err != nil {
		return nil, err
	}
	var hour, minute, second int
	if spec.From != "" {
		from := strings.Split(spec.From, ":")
		var fields []int
		for _, elem := range from {
			t, err := strconv.Atoi(elem)
			if err != nil {
				return nil, err
			}
			fields = append(fields, t)
		}
		switch len(fields) {
		case 2:
			hour, minute = fields[0], fields[1]
		case 3:
			hour, minute, second = fields[0], fields[1], fields[2]
		default:
			return nil, fmt.Errorf("invalid spec field: %s", spec.From)
		}
	}
	return &schedule{duration: duration, hour: hour, minute: minute, second: second}, nil
}

func (s *schedule) next(prev time.Time) time.Time {
	origin := time.Date(prev.Year(), prev.Month(), prev.Day(), s.hour, s.minute, s.second, 0, prev.Location())
	diff := prev.Sub(origin)
	if diff >= 0 {
		diff = (diff/s.duration + 1) * s.duration
	} else {
		diff = (diff / s.duration) * s.duration
	}
	return origin.Add(diff)
}
