package cron

import (
	"testing"
	"time"

	"github.com/stretchr/testify/suite"
)

type ParserTestSuite struct {
	suite.Suite
	testcases []*parserTestcase
}

type parserTestcase struct {
	spec        *Spec
	parseFailed bool
	schedule    *schedule
	next        map[string]string
}

func (suite *ParserTestSuite) SetupSuite() {
	suite.testcases = []*parserTestcase{
		{
			spec:     &Spec{Every: "1h"},
			schedule: &schedule{duration: time.Hour},
			next:     map[string]string{"2006-01-02T15:04:05Z": "2006-01-02T16:00:00Z"},
		},
		{
			spec:        &Spec{Every: "1a"},
			parseFailed: true,
		},
		{
			spec: &Spec{Every: "24h", From: "00:05:00"},
			schedule: &schedule{
				duration: 24 * time.Hour,
				minute:   5,
			},
			next: map[string]string{
				"2021-12-31T15:04:05Z": "2022-01-01T00:05:00Z",
				"2022-01-01T00:05:00Z": "2022-01-02T00:05:00Z",
			},
		},
		{
			spec: &Spec{Every: "6h", From: "00:00:00"},
			schedule: &schedule{
				duration: 6 * time.Hour,
			},
			next: map[string]string{
				"2021-12-31T15:04:05Z": "2021-12-31T18:00:00Z",
				"2021-12-31T18:00:00Z": "2022-01-01T00:00:00Z",
				"2022-01-01T08:00:00Z": "2022-01-01T12:00:00Z",
			},
		},
	}
}

func (suite *ParserTestSuite) TestParse() {
	for _, c := range suite.testcases {
		sched, err := parse(c.spec)
		if c.parseFailed {
			suite.Require().Error(err)
			continue
		}
		suite.Require().NoError(err)
		suite.Equal(c.schedule, sched)
	}
}

func (suite *ParserTestSuite) TestSchedule_Next() {
	for _, c := range suite.testcases {
		if !c.parseFailed {
			for now, next := range c.next {
				t, err := time.Parse(time.RFC3339, now)
				suite.Require().NoError(err)
				expected, err := time.Parse(time.RFC3339, next)
				suite.Require().NoError(err)
				got := c.schedule.next(t)
				suite.Equal(expected, got)
			}
		}
	}
}

func TestParser(t *testing.T) {
	suite.Run(t, new(ParserTestSuite))
}
