package cron

import (
	"context"
	"sort"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
)

type job struct {
	fn       func(ctx context.Context)
	nextTime time.Time
	schedule *schedule
}

type Scheduler struct {
	ctx        context.Context
	cancelFunc context.CancelFunc
	wg         *sync.WaitGroup
	jobs       []*job
	logger     log.Logger
}

func NewScheduler(logger log.Logger) *Scheduler {
	ctx, cancel := context.WithCancel(context.Background())
	return &Scheduler{logger: logger, ctx: ctx, cancelFunc: cancel, wg: &sync.WaitGroup{}}
}

func (s *Scheduler) Add(name string, spec *Spec, fn JobFunc, d ...DecoratorFunc) error {
	schedule, err := parse(spec)
	if err != nil {
		return err
	}

	for i := len(d); i > 0; i-- {
		fn = d[i-1](fn)
	}
	s.jobs = append(s.jobs, &job{
		schedule: schedule,
		fn: func(ctx context.Context) {
			defer func() {
				if err := recover(); err != nil {
					s.logger.Errorf("%s job: recover from panic: %v", name, err)
				}
			}()
			if _, err := fn(ctx); err != nil && err != context.Canceled {
				s.logger.Errorf("%s job: %v", name, err)
			}
		},
	})
	return nil
}

func (s *Scheduler) Start() {
	loc, err := time.LoadLocation("Europe/Moscow")
	if err != nil {
		panic(err)
	}
	if len(s.jobs) == 0 {
		return
	}
	now := time.Now().In(loc)
	for _, j := range s.jobs {
		j.nextTime = j.schedule.next(now)
	}
	go func() {
		for {
			sort.Slice(s.jobs, func(i, j int) bool { return s.jobs[i].nextTime.Unix() < s.jobs[j].nextTime.Unix() })
			timer := time.NewTimer(s.jobs[0].nextTime.Sub(now))
			select {
			case now = <-timer.C:
				s.wg.Add(1)
				nextFunc := s.jobs[0].fn
				s.jobs[0].nextTime = s.jobs[0].schedule.next(now)
				ctx := context.WithValue(s.ctx, nextTimeKey{}, s.jobs[0].nextTime)
				go func(ctx context.Context) {
					defer s.wg.Done()
					nextFunc(ctx)
				}(ctx)

			case <-s.ctx.Done():
				return
			}
		}
	}()
}

func (s *Scheduler) Stop() {
	s.cancelFunc()
	s.wg.Wait()
}

type nextTimeKey struct{}

func NextTimeFromContext(ctx context.Context) *time.Time {
	if t, ok := ctx.Value(nextTimeKey{}).(time.Time); ok {
		return &t
	}
	return nil
}
