package db

import (
	"context"
	"crypto/tls"
	"crypto/x509"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"strings"
	"time"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"
	"go.mongodb.org/mongo-driver/tag"
	"go.mongodb.org/mongo-driver/x/bsonx/bsoncore"
)

type MongoGeoTags map[string]string

type MongoConfig struct {
	Username    string       `yaml:"username"`
	Password    string       `yaml:"password"`
	Hosts       []string     `yaml:"hosts"`
	Database    string       `yaml:"database"`
	ReplicaSet  string       `yaml:"replica_set,omitempty"`
	MinPoolSize uint64       `yaml:"min_pool_size,omitempty"`
	MaxPoolSize uint64       `yaml:"max_pool_size,omitempty"`
	TLS         string       `yaml:"tls_ca_file"`
	GeoTags     MongoGeoTags `yaml:"geo_tags,omitempty"`
}

func NewMongoConnect(ctx context.Context, config MongoConfig) (*mongo.Database, error) {
	if len(config.Hosts) == 0 {
		return nil, errors.New("no hosts in config")
	}
	auth := options.Credential{
		Username:   config.Username,
		Password:   config.Password,
		AuthSource: config.Database,
	}
	opts := options.Client().
		SetAuth(auth).
		SetHosts(config.Hosts).
		SetReplicaSet(config.ReplicaSet).
		SetMinPoolSize(config.MinPoolSize)
	if config.MaxPoolSize != 0 {
		opts = opts.SetMaxPoolSize(config.MaxPoolSize)
	}
	if config.TLS != "" {
		tlsConfig := &tls.Config{}
		data, err := ioutil.ReadFile(config.TLS)
		if err != nil {
			return nil, err
		}
		tlsConfig.RootCAs = x509.NewCertPool()
		if !tlsConfig.RootCAs.AppendCertsFromPEM(data) {
			return nil, errors.New("the specified CA file does not contain any valid certificates")
		}
		opts.SetTLSConfig(tlsConfig)
	}

	client, err := mongo.Connect(ctx, opts)
	if err != nil {
		return nil, err
	}

	pingCtx, pingCancel := context.WithTimeout(ctx, 5*time.Second)
	defer pingCancel()

	return client.Database(config.Database), client.Ping(pingCtx, nil)
}

func NewMongoLocalReadPreference(tags MongoGeoTags) *readpref.ReadPref {
	return readpref.SecondaryPreferred(readpref.WithTagSets(tag.NewTagSetFromMap(tags), tag.Set{}))
}

func GetTestingMongoDB() (*mongo.Database, error) {
	port := os.Getenv("RECIPE_MONGO_PORT")
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	client, err := mongo.Connect(ctx, options.Client().ApplyURI(fmt.Sprintf("mongodb://localhost:%s", port)))
	if err != nil {
		return nil, err
	}

	return client.Database("testing"), nil
}

type mongoSelectionKey struct {
	orderNumber int
	subKeys     []string
}

type MongoSelection struct {
	cursor *mongo.Cursor
	ctx    context.Context
	keys   map[string]mongoSelectionKey
}

func NewMongoSelection(
	ctx context.Context,
	collection *mongo.Collection,
	filter bson.D,
	opts *options.FindOptions,
	keys []string,
) (*MongoSelection, error) {
	if len(keys) == 0 {
		return nil, fmt.Errorf("no field selected")
	}
	projection := bson.D{}
	selectionKeys := make(map[string]mongoSelectionKey)
	for i, key := range keys {
		projection = append(projection, bson.E{Key: key, Value: 1})
		selectionKeys[key] = mongoSelectionKey{orderNumber: i, subKeys: strings.Split(key, ".")}
	}
	if _, ok := selectionKeys["_id"]; !ok {
		projection = append(projection, bson.E{Key: "_id", Value: 0})
	}
	opts = opts.SetProjection(projection)
	cursor, err := collection.Find(ctx, filter, opts)
	if err != nil {
		return nil, err
	}
	return &MongoSelection{
		cursor: cursor,
		ctx:    ctx,
		keys:   selectionKeys,
	}, nil
}

func (selection *MongoSelection) Next() bool {
	return selection.cursor.Next(selection.ctx)
}

func (selection *MongoSelection) Scan(args ...interface{}) error {
	if len(args) == 0 || len(args) != len(selection.keys) {
		return fmt.Errorf("wrong args number")
	}
	var raw bson.Raw
	if err := selection.cursor.Decode(&raw); err != nil {
		return err
	}
	for _, key := range selection.keys {
		val, err := raw.LookupErr(key.subKeys...)
		if err == bsoncore.ErrElementNotFound {
			continue
		}
		if err := val.Unmarshal(args[key.orderNumber]); err != nil {
			return err
		}
	}
	return nil
}

func (selection *MongoSelection) Close() error {
	return selection.cursor.Close(selection.ctx)
}
