package db

import (
	"context"
	"crypto/tls"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/auth"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
)

type YDBConfig struct {
	Endpoint string `yaml:"endpoint"`
	Database string `yaml:"database"`
	Token    string `yaml:"token"`
}

type YDBClient struct {
	client         *table.Client
	pool           *table.SessionPool
	database       string
	ReadTxControl  *table.TransactionControl
	WriteTxControl *table.TransactionControl
}

func NewYDBClient(ctx context.Context, config YDBConfig) (*YDBClient, error) {
	var credentials ydb.Credentials
	var tlsConfig *tls.Config
	switch {
	case config.Token == "":
		var err error
		if credentials, err = auth.FromEnviron(context.Background()); err != nil {
			return nil, err
		}
		tlsConfig = &tls.Config{}
	default:
		credentials = ydb.AuthTokenCredentials{
			AuthToken: config.Token,
		}
	}
	dialer := &ydb.Dialer{
		DriverConfig: &ydb.DriverConfig{
			Database:    config.Database,
			Credentials: credentials,
		},
		Timeout:   5 * time.Second,
		TLSConfig: tlsConfig,
	}

	driver, err := dialer.Dial(ctx, config.Endpoint)
	if err != nil {
		return nil, err
	}

	tc := &table.Client{
		Driver: driver,
	}
	pool := &table.SessionPool{
		IdleThreshold: 5 * time.Second,
		Builder:       tc,
	}

	return &YDBClient{
		client:         tc,
		pool:           pool,
		database:       config.Database,
		ReadTxControl:  table.TxControl(table.BeginTx(table.WithOnlineReadOnly()), table.CommitTx()),
		WriteTxControl: table.TxControl(table.BeginTx(table.WithSerializableReadWrite()), table.CommitTx()),
	}, nil
}

func (client *YDBClient) GetPath(tableName string) string {
	return fmt.Sprintf("%s/%s", client.database, tableName)
}

func (client *YDBClient) Retry(ctx context.Context, fn func(c context.Context, s *table.Session) error) error {
	return table.Retry(ctx, client.pool, table.OperationFunc(fn))
}

func (client *YDBClient) ExecuteReadQuery(ctx context.Context, query string, opts ...table.ParameterOption) (res *table.Result, err error) {
	err = client.Retry(ctx, func(c context.Context, s *table.Session) error {
		_, res, err = s.Execute(c, client.ReadTxControl, query, table.NewQueryParameters(opts...), EnableQueryCache())
		return err
	})
	return
}

func (client *YDBClient) ExecuteWriteQuery(ctx context.Context, query string, opts ...table.ParameterOption) error {
	return client.Retry(ctx, func(c context.Context, s *table.Session) error {
		_, _, err := s.Execute(c, client.WriteTxControl, query, table.NewQueryParameters(opts...), EnableQueryCache())
		return err
	})
}

func (client *YDBClient) BulkUpsert(ctx context.Context, name string, rows ydb.Value) error {
	return client.Retry(ctx, func(c context.Context, s *table.Session) error {
		return s.BulkUpsert(c, client.GetPath(name), rows)
	})
}

func (client *YDBClient) Close() error {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	return client.pool.Close(ctx)
}

func (client *YDBClient) CreateSession(ctx context.Context) (*table.Session, error) {
	return client.client.CreateSession(ctx)
}

func EnableQueryCache() table.ExecuteDataQueryOption {
	return table.WithQueryCachePolicy(table.WithQueryCachePolicyKeepInCache())
}
