package httputil

import (
	"fmt"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
)

// ZapLoggerMiddleware is an example of echo middleware that logs requests using logger "zap"
func ZapLoggerMiddleware(logger log.Logger) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			start := time.Now()
			err := next(c)
			if err != nil {
				c.Error(err)
			}

			req := c.Request()
			res := c.Response()

			id := req.Header.Get(echo.HeaderXRequestID)
			if id == "" {
				id = res.Header().Get(echo.HeaderXRequestID)
			}

			fields := []log.Field{
				log.Int("status", res.Status),
				log.String("latency", time.Since(start).String()),
				log.String("id", id),
				log.String("method", req.Method),
				log.String("uri", req.RequestURI),
				log.String("host", req.Host),
				log.String("remote_ip", c.RealIP()),
			}

			msgFields := []interface{}{
				req.Method,
				req.Host,
				req.RequestURI,
				res.Status,
				time.Since(start).String(),
				c.RealIP(),
			}

			msg := fmt.Sprintf("%s %s%s %d %s %s", msgFields...)

			n := res.Status
			switch {
			case n >= 500:
				logger.Error(msg, fields...)
			case n >= 400:
				logger.Warn(msg, fields...)
			default:
				logger.Info(msg, fields...)
			}

			return nil
		}
	}
}

func RespondError(ctx echo.Context, status int, err error, prefix string) error {
	msg := err.Error() + "\n"
	if prefix != "" {
		msg = fmt.Sprintf("%s:%s", prefix, msg)
	}
	switch {
	case status >= 500:
		ctx.Logger().Error(msg)
	case status >= 400:
		ctx.Logger().Warn(msg)
	}
	return ctx.String(status, msg)
}
