package httputil

import (
	"context"
	"crypto/tls"
	"net/http"
	"time"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
)

type TLSConfig struct {
	KeyFile  string `yaml:"key_file"`
	CertFile string `yaml:"cert_file"`
}

type ServerConfig struct {
	Address      string     `yaml:"address"`
	ReadTimeout  string     `yaml:"read_timeout,omitempty"`
	WriteTimeout string     `yaml:"write_timeout,omitempty"`
	TLS          *TLSConfig `yaml:"tls,omitempty"`
}

func NewServer(config ServerConfig, handler http.Handler) (*http.Server, error) {
	if config.ReadTimeout == "" {
		config.ReadTimeout = "0s"
	}
	readTimeout, err := time.ParseDuration(config.ReadTimeout)
	if err != nil {
		return nil, err
	}
	if config.WriteTimeout == "" {
		config.WriteTimeout = "0s"
	}
	writeTimeout, err := time.ParseDuration(config.WriteTimeout)
	if err != nil {
		return nil, err
	}
	server := &http.Server{
		Addr:         config.Address,
		ReadTimeout:  readTimeout,
		WriteTimeout: writeTimeout,
		Handler:      handler,
	}
	if config.TLS != nil {
		server.TLSConfig = &tls.Config{}
	}
	return server, nil
}

func NewServeFunc(config ServerConfig, server *http.Server) lib.ServeFunc {
	return func(ctx context.Context) {
		errCh := make(chan error)
		go func() {
			if config.TLS != nil {
				errCh <- server.ListenAndServeTLS(config.TLS.CertFile, config.TLS.KeyFile)
			} else {
				errCh <- server.ListenAndServe()
			}
		}()
		select {
		case <-ctx.Done():
			_ = server.Shutdown(context.Background())
		case err := <-errCh:
			panic(err)
		}
	}
}
