package juggler

// Juggler client delay (few seconds)
// + client-side or server-side batching (up to 10 seconds)
// + aggregator period (90 sec)
// + banshee delay (few seconds)
// + push sender batch collection time (5 seconds)
const jugglerPassthroughTime = 2 * 60

// Wall-E configures notifications for host with this period for repeat data
const jugglerPushPeriod = 5 * 60

func probablyEffectiveTimestamp(check JugglerChild, statusMtime int64, receiveTimestamp int64) int64 {
	// Calculate probable check effective timestamp
	// (script start time for passive checks, first ping for active checks).
	//
	// Try use status mtime as more accurate representation of an event time.
	// Use receive timestamp as more recent representation.

	period := check.GetPeriod()
	statusMtimeBased := statusMtime
	periodical := receiveTimestamp - jugglerPushPeriod - jugglerPassthroughTime - 2*period
	return max(statusMtimeBased, periodical)
}

func calculateEffectiveTimestamp(
	check JugglerChild,
	status WalleStatus,
	statusMtime int64,
	receiveTimestamp int64,
) (int64, error) {
	if status == WalleCheckStatusMissing || status == WalleCheckStatusInvalid {
		return receiveTimestamp, nil
	}
	parsedMetadata, err := check.ParseMetadata()
	if err != nil {
		return 0, err
	}
	accuracy := check.GetAccuracy()
	var timestamp int64
	switch {
	case check.ServiceName == CheckTypeMemory:
		timestamp = int64(parsedMetadata.Results.Mem.Timestamp)
	case check.HasRootMetadataTimestamp():
		// this is a passive check from automation plot. It must have timestamp in its metadata.
		timestamp = int64(parsedMetadata.Timestamp)
		if timestamp == 0 {
			timestamp = int64(parsedMetadata.Result.Timestamp)
		}
		if timestamp == 0 {
			timestamp = int64(parsedMetadata.Results.Timestamp)
		}
	case check.IsHwWatcherCheck():
		// Only hw watcher checks have timestamp built into metadata (currently).
		timestamp = int64(parsedMetadata.Result.Timestamp)
		if timestamp == 0 {
			timestamp = int64(parsedMetadata.Results.Timestamp)
		}
	default:
		timestamp = probablyEffectiveTimestamp(check, statusMtime, receiveTimestamp)
	}
	if timestamp == 0 {
		return 0, errMetadataTimestampNotFound
	}
	return timestamp - accuracy, nil
}

var allAccuracy = initAccuracyMap()
var allPeriod = initPeriodMap()

// Maximum delay between check script start moment and actual check data produced.
// It is an actual timeout used in juggler-client to terminate checks scripts that run too long.
const passiveCheckAccuracy int64 = 3 * 60

func initAccuracyMap() map[CheckType]int64 {
	result := make(map[CheckType]int64)
	const unreachAccuracy int64 = 60 // This is a pongerd built-in flap detection
	result[CheckTypeUnreachable] = unreachAccuracy
	result[CheckTypeSSH] = 1 // Assume check timeout as it's accuracy, 1 sec for ssh check

	// hw-watcher check time is a time when the check has been finished. Check running time varies: it may be a few
	// seconds for link check or up to 3 minutes for disk check when there are too many disks or when check times out on BOT
	// API.
	const hwWatcherCheckAccuracy int64 = 3 * 60
	for _, checkType := range jugglerAllCheckTypesHwWatcher {
		result[checkType] = hwWatcherCheckAccuracy
	}

	for _, checkType := range jugglerAllCheckTypesNonHwWatcherPassive {
		// hw-watcher check time is a time when the check has been finished. Check running time varies: it may be a few
		// seconds for link check or up to 3 minutes for disk check when there are too many disks or when check times out on BOT
		// API.
		result[checkType] = passiveCheckAccuracy
	}

	result[CheckTypeRack] = unreachAccuracy
	result[CheckTypeRackOverheat] = hwWatcherCheckAccuracy

	// We check netmon connectivity for DC, not for switch:
	// * switch takes 30 minutes to collect the data, we can not afford it.
	// * dc level contains breakdown up to the switch level, it is just less accurate, so we can get all data we need.
	// * we have to mark switch as failing as soon as it's connectivity drops to 90% to react faster.
	// That means, we'll get some false OKs as well as some false FAILs.
	result[CheckTypeNetmon] = 2 * 60

	return result
}

// Juggler passive check (running on Juggler client) period
const passiveCheckPeriod = 5 * 60

func initPeriodMap() map[CheckType]int64 {
	result := make(map[CheckType]int64)

	// Juggler UNREACHABLE check doesn't have a period
	// it has a pongerd built-in flap detection and probes every 5 seconds.
	const unreachPeriod int64 = 5

	result[CheckTypeUnreachable] = unreachPeriod
	result[CheckTypeSSH] = 60 // Juggler ssh check runs every 1.5 minute with timeout 1 sec

	// Juggler passive check (running on Juggler client) period
	const passiveCheckPeriod = 5 * 60
	var allPassiveChecks []CheckType
	allPassiveChecks = append(allPassiveChecks, jugglerAllCheckTypesHwWatcher[:]...)
	allPassiveChecks = append(allPassiveChecks, jugglerAllCheckTypesNonHwWatcherPassive[:]...)
	for _, checkType := range allPassiveChecks {
		result[checkType] = passiveCheckPeriod
	}

	result[CheckTypeRack] = unreachPeriod
	result[CheckTypeRackOverheat] = passiveCheckPeriod

	// Period for polling netmon
	result[CheckTypeNetmon] = 2 * 60

	result[CheckTypeNeedRebootKernel] = 30 * 60
	result[CheckTypeFirmware] = 24 * 60 * 60

	//NOTE(rocco66): linkeye pushes new raw event to juggler every 30 min
	result[CheckTypeIbLink] = 30 * 60

	return result
}
