package juggler

import (
	"encoding/json"
	"errors"
	"fmt"

	"github.com/labstack/echo/v4"
)

const DefaultHostsCountInRequest = 100

type CheckType string

const CheckTypeUnreachable CheckType = "UNREACHABLE"
const CheckTypeSSH CheckType = "ssh"
const CheckTypeMeta CheckType = "META"
const CheckTypeWalleMeta CheckType = "walle_meta"
const CheckTypeReboots CheckType = "walle_reboots"
const CheckTypeTaintedKernel CheckType = "walle_tainted_kernel"
const CheckTypeCPU CheckType = "walle_cpu"
const CheckTypeCPUCaches CheckType = "walle_cpu_caches"
const CheckTypeCPUCapping CheckType = "walle_cpu_capping"
const CheckTypeRack CheckType = "walle_rack"
const CheckTypeRackOverheat CheckType = "walle_rack_overheat"
const CheckTypeFSCheck CheckType = "walle_fs_check"
const CheckTypeClocksource CheckType = "walle_clocksource"
const CheckTypeTorLink CheckType = "walle_tor_link"
const CheckTypeIbLink CheckType = "walle_ib_link"
const CheckTypeMemory CheckType = "walle_memory"
const CheckTypeDisk CheckType = "walle_disk"
const CheckTypeLink CheckType = "walle_link"
const CheckTypeBmc CheckType = "walle_bmc"
const CheckTypeGpu CheckType = "walle_gpu"
const CheckTypeInfiniband CheckType = "walle_infiniband"
const CheckTypeNetmon CheckType = "netmon"
const CheckTypeNeedRebootKernel CheckType = "need_reboot_kernel"
const CheckTypeFirmware CheckType = "walle_firmware"

var jugglerAllCheckTypesActive = [...]CheckType{
	CheckTypeUnreachable,
	CheckTypeSSH,
}

var jugglerAllCheckTypesNonHwWatcherPassive = [...]CheckType{
	CheckTypeMeta,
	CheckTypeWalleMeta,
	CheckTypeReboots,
	CheckTypeTaintedKernel,
	CheckTypeCPU,
	CheckTypeCPUCapping,
	CheckTypeFSCheck,
	CheckTypeTorLink,
	CheckTypeIbLink,
}

var jugglerAllCheckTypesHwWatcher = [...]CheckType{
	CheckTypeMemory,
	CheckTypeDisk,
	CheckTypeLink,
	CheckTypeBmc,
	CheckTypeGpu,
	CheckTypeCPUCaches,
	CheckTypeGpu,
	CheckTypeInfiniband,
}
var jugglerAllCheckTypesWalleBased = [...]CheckType{
	CheckTypeRack,
	CheckTypeRackOverheat,
}

const checkFilterActual = "actual"
const checkFilterInvalid = "invalid"
const checkFilterFlapping = "flapping"
const checkFilterDowntime = "downtime"
const checkFilterUnreach = "unreach"
const checkFilterForceOk = "force_ok"
const CheckFilterNodata = "no_data"

var jugglerCheckFiltersActual = [...]string{checkFilterActual}
var jugglerCheckFiltersActualForceOk = [...]string{checkFilterActual, checkFilterForceOk}
var jugglerCheckFiltersActualFlapping = [...]string{checkFilterActual, checkFilterFlapping}
var jugglerCheckFiltersActualMissing = [...]string{checkFilterActual, CheckFilterNodata}
var jugglerCheckFiltersAllMissing = [...]string{checkFilterInvalid, CheckFilterNodata}

type JugglerChildActual struct {
	Status      JugglerStatus `json:"status"`
	StatusMtime int64         `json:"status_mtime"`
	Metadata    string        `json:"description"`
}

const JugglerCrit = "CRIT"
const JugglerWarn = "WARN"
const JugglerInfo = "INFO"
const JugglerOk = "OK"

type WalleStatus string

const WalleCheckStatusPassed = WalleStatus("passed")

// Host marked suspected because check failed for host but it's to early to mark it failed.
const WalleCheckStatusSuspected = WalleStatus("suspected")

// Check data is missing - Wall-E has received a NO DATA check.
const WalleCheckStatusMissing = WalleStatus("missing")

// Check has invalid metadata.
const WalleCheckStatusInvalid = WalleStatus("invalid")

// Host marked failed because check failed.
const WalleCheckStatusFailed = WalleStatus("failed")

var jugglerToWalleStatuses = map[JugglerStatus]WalleStatus{
	JugglerOk:   WalleCheckStatusPassed,
	JugglerInfo: WalleCheckStatusSuspected,
	JugglerWarn: WalleCheckStatusSuspected,
	JugglerCrit: WalleCheckStatusFailed,
}

var allMetachecks map[CheckType]bool
var allHwWatcherChecks map[CheckType]bool
var allKnownChecks map[CheckType]bool
var allPassiveCheckTypes map[CheckType]struct{}

func init() {
	result := make(map[CheckType]bool)
	for _, rackCheckType := range RackCheckMapping {
		result[rackCheckType.WalleName] = true
	}
	allMetachecks = result
	result = make(map[CheckType]bool)
	for _, t := range jugglerAllCheckTypesHwWatcher {
		result[t] = true
	}
	allHwWatcherChecks = result
	result = make(map[CheckType]bool)
	var knownChecks []CheckType
	knownChecks = append(knownChecks, jugglerAllCheckTypesActive[:]...)
	knownChecks = append(knownChecks, jugglerAllCheckTypesNonHwWatcherPassive[:]...)
	knownChecks = append(knownChecks, jugglerAllCheckTypesHwWatcher[:]...)
	knownChecks = append(knownChecks, jugglerAllCheckTypesWalleBased[:]...)
	knownChecks = append(knownChecks, CheckTypeNetmon)
	for _, checkType := range knownChecks {
		result[checkType] = true
	}
	allKnownChecks = result

	allPassiveCheckTypes = make(map[CheckType]struct{})
	for _, t := range jugglerAllCheckTypesHwWatcher {
		allPassiveCheckTypes[t] = struct{}{}
	}
	for _, t := range jugglerAllCheckTypesNonHwWatcherPassive {
		allPassiveCheckTypes[t] = struct{}{}
	}
}

type JugglerStatus string

func (jugglerStatus JugglerStatus) ToWalleStatus() WalleStatus {
	return jugglerToWalleStatuses[jugglerStatus]
}

type JugglerChild struct {
	HostName     string             `json:"host_name"`
	ServiceName  CheckType          `json:"service_name"`
	InstanceName string             `json:"instance_name"`
	Status       JugglerStatus      `json:"status"` // TODO(rocco66): do we need it and Actual.Status both?
	Flags        []string           `json:"flags"`
	Actual       JugglerChildActual `json:"actual"`
}

func (child *JugglerChild) AnyChanges(status WalleStatus, statusMtime int64, metadata string) bool {
	return child.Status.ToWalleStatus() != status ||
		child.Actual.StatusMtime != statusMtime ||
		child.Actual.Metadata != metadata
}

func (child *JugglerChild) IsWalleMetaChecks() bool {
	// Meta-checks generated ny Wall-E.
	return allMetachecks[child.ServiceName]
}

func (child *JugglerChild) IsHwWatcherCheck() bool {
	return allHwWatcherChecks[child.ServiceName]
}

func (child *JugglerChild) HasRootMetadataTimestamp() bool {
	return child.isAutomationPlotPassiveCheck() ||
		child.ServiceName == CheckTypeFirmware ||
		child.ServiceName == CheckTypeBmc ||
		child.ServiceName == CheckTypeIbLink
}

func (child *JugglerChild) isAutomationPlotPassiveCheck() bool {
	return !allKnownChecks[child.ServiceName]
}

func (child *JugglerChild) ContainFlag(target string) bool {
	for _, flag := range child.Flags {
		if flag == target {
			return true
		}
	}
	return false
}

func (child *JugglerChild) IsMissing() bool {
	for _, missingFlag := range jugglerCheckFiltersAllMissing {
		if child.ContainFlag(missingFlag) {
			return true
		}
	}
	return false
}

func (child *JugglerChild) Key() WalleCheckKey {
	return WalleCheckKey(MkCheckKey(child.HostName, child.ServiceName))
}

func (child *JugglerChild) GetAccuracy() int64 {
	if res, ok := allAccuracy[child.ServiceName]; ok {
		return res
	}
	return passiveCheckAccuracy
}

func (child *JugglerChild) ParseMetadata() (*ParsedMetadata, error) {
	result := &ParsedMetadata{}
	var err error
	if child.metadataIsJSON() {
		err = json.Unmarshal([]byte(child.Actual.Metadata), result)
	}
	return result, err
}
func (child *JugglerChild) metadataIsJSON() bool {
	return !(child.ServiceName == CheckTypeUnreachable ||
		child.ServiceName == CheckTypeSSH ||
		child.ServiceName == CheckTypeMeta)
}

func (child *JugglerChild) GetPeriod() int64 {
	if res, ok := allPeriod[child.ServiceName]; ok {
		return res
	}
	return passiveCheckPeriod
}

func MkCheckKey(host string, service CheckType) string {
	return fmt.Sprintf("%s|%s", host, service)
}

type WalleCheckKey string

type AggrCheckFromRequest struct {
	HostName    string         `json:"host_name"`
	ServiceName CheckType      `json:"service_name"`
	Children    []JugglerChild `json:"children"`
}

type HealthRequest struct {
	Checks []AggrCheckFromRequest `json:"checks"`
}

var errMetadataTimestampNotFound = errors.New("passive check metadata not found error")

type ParsedMetadata struct {
	Results   metaDataResults `json:"results"`
	Result    metaDataResult  `json:"result"`
	Timestamp float64         `json:"timestamp"`
}

type metaDataResults struct {
	Mem       metaDataMem `json:"mem"`
	Timestamp float64     `json:"timestamp"`
}

type metaDataMem struct {
	Timestamp float64 `json:"timestamp"`
}

type metaDataResult struct {
	Timestamp float64 `json:"timestamp"`
}

type GroupedByHostsChecks map[string][]JugglerChild

func (checks GroupedByHostsChecks) AddChild(
	children []JugglerChild,
	checkKeys []WalleCheckKey,
) []WalleCheckKey {
	for _, child := range children {
		checks[child.HostName] = append(checks[child.HostName], child)
		checkKeys = append(checkKeys, child.Key())
	}
	return checkKeys
}

func ReshapeChecks(store RackTopologyStore, request *HealthRequest, logger echo.Logger) (
	GroupedByHostsChecks,
	[]WalleCheckKey,
) {
	groupedChecks := make(GroupedByHostsChecks, DefaultHostsCountInRequest)
	checkKeys := make([]WalleCheckKey, 0, DefaultHostsCountInRequest)
	Metrics.BatchAggrCount.Update(float64(len(request.Checks)))
	var checksReceived int
	for _, requestAggrCheck := range request.Checks {
		Metrics.ReceivedEvents.Update(float64(len(requestAggrCheck.Children)))
		checksReceived += len(requestAggrCheck.Children)
		if mappedRackCheck, ok := RackCheckMapping[requestAggrCheck.ServiceName]; ok {
			rackStatus := GetRackStatus(requestAggrCheck.Children, mappedRackCheck)
			if rackStatus.JugglerStatus == JugglerCrit || rackStatus.TooSmall() {
				// if whole rack is broken or it is too small, we don't need to split it
				checkKeys = groupedChecks.AddChild(rackStatus.ReshapedChildren, checkKeys)
			} else {
				// split rack half-rack might be broken
				rackName := requestAggrCheck.HostName
				splittedChecksList := splitHalfrack(store, rackName, requestAggrCheck.Children)
				for _, splittedChecks := range splittedChecksList {
					rackStatus := GetRackStatus(splittedChecks, mappedRackCheck)
					checkKeys = groupedChecks.AddChild(rackStatus.ReshapedChildren, checkKeys)
				}
			}
		}
		if requestAggrCheck.ServiceName == CheckTypeIbLink {
			aggregatedInfinibandLinkChecks := AggregateInfinibandPortChecks(requestAggrCheck.Children, logger)
			checkKeys = groupedChecks.AddChild(aggregatedInfinibandLinkChecks, checkKeys)
		} else {
			checkKeys = groupedChecks.AddChild(requestAggrCheck.Children, checkKeys)
		}
	}
	Metrics.BatchChecksCount.Update(float64(checksReceived))
	return groupedChecks, checkKeys
}

func AllCheckTypes() []CheckType {
	types := make([]CheckType, 0, len(allKnownChecks))
	for k := range allKnownChecks {
		types = append(types, k)
	}
	return types
}

func IsActiveCheckType(checkType CheckType) bool {
	return checkType == CheckTypeUnreachable || checkType == CheckTypeSSH
}

func IsMetaCheckType(checkType CheckType) bool {
	return checkType == CheckTypeMeta || checkType == CheckTypeWalleMeta
}

func IsPassiveCheckType(checkType CheckType) bool {
	_, ok := allPassiveCheckTypes[checkType]
	return ok
}

func IsWalleCheckType(checkType CheckType) bool {
	return checkType == CheckTypeRack || checkType == CheckTypeRackOverheat
}

func IsNetmonCheckType(checkType CheckType) bool {
	return checkType == CheckTypeNetmon
}

func AllWalleStatuses() []WalleStatus {
	return []WalleStatus{
		WalleCheckStatusPassed,
		WalleCheckStatusFailed,
		WalleCheckStatusInvalid,
		WalleCheckStatusMissing,
		WalleCheckStatusSuspected,
	}
}
