package juggler

import (
	"encoding/json"

	"github.com/labstack/echo/v4"
)

const TypicalInfinibandPortsCount = 4

type hostChecks struct {
	PassedChecks    []JugglerChild
	FailedChecks    []JugglerChild
	MissedChecks    []JugglerChild
	SuspectedChecks []JugglerChild
}

func (checks *hostChecks) getAggregatedMetadata(reason string, timestamp int64) string {
	newMetadata, _ := json.Marshal(AggregatedMetadata{
		timestamp,
		reason,
		stripChecks(checks.PassedChecks),
		stripChecks(checks.FailedChecks),
		stripChecks(checks.MissedChecks),
		stripChecks(checks.SuspectedChecks),
	})
	return string(newMetadata)
}

func (checks *hostChecks) GetAggregatedCheck() JugglerChild {
	var targetChecks []JugglerChild
	var status JugglerStatus
	var flags []string
	var reason string
	switch {
	case len(checks.FailedChecks) > 0:
		targetChecks = checks.FailedChecks
		status = JugglerCrit
		flags = jugglerCheckFiltersActual[:]
		reason = "Some ib_link checks are failed"
	case len(checks.SuspectedChecks) > 0:
		targetChecks = checks.SuspectedChecks
		status = JugglerWarn
		flags = jugglerCheckFiltersActual[:]
		reason = "Some ib_link checks are suspected"
	case len(checks.MissedChecks) > 0:
		targetChecks = checks.MissedChecks
		status = JugglerWarn
		flags = jugglerCheckFiltersActualMissing[:]
		reason = "Some ib_link checks are missing"
	default:
		targetChecks = checks.PassedChecks
		status = JugglerOk
		flags = jugglerCheckFiltersActual[:]
		reason = "No errs"
	}
	aggregatedStatusMtime, aggregatedTimestamp := findTimes(targetChecks)
	if aggregatedTimestamp == 0 {
		aggregatedTimestamp = aggregatedStatusMtime
	}
	return JugglerChild{
		targetChecks[0].HostName,
		targetChecks[0].ServiceName,
		"",
		status,
		flags,
		JugglerChildActual{
			status,
			aggregatedStatusMtime,
			checks.getAggregatedMetadata(reason, aggregatedTimestamp),
		},
	}
}

type rawCheckMetadata struct {
	Timestamp float64 `json:"timestamp"`
}

func getRawTimestamp(child *JugglerChild) int64 {
	var rawMetadata rawCheckMetadata
	err := json.Unmarshal([]byte(child.Actual.Metadata), &rawMetadata)
	if err == nil {
		return int64(rawMetadata.Timestamp)
	}
	return 0
}

func findTimes(children []JugglerChild) (int64, int64) {
	if len(children) == 0 {
		return 0, 0
	}
	oldestStatusTime := &children[0]
	oldestTimestamp := getRawTimestamp(&children[0])
	for childIndex := 1; childIndex < len(children)-1; childIndex++ {
		child := &children[childIndex]
		if child.Actual.StatusMtime < oldestStatusTime.Actual.StatusMtime {
			oldestStatusTime = child
		}
		ts := getRawTimestamp(child)
		if ts < oldestTimestamp {
			oldestTimestamp = ts
		}
	}
	return oldestStatusTime.Actual.StatusMtime, oldestTimestamp
}

type AggregatedMetadataCheck struct {
	Port        string        `json:"port"`
	Status      JugglerStatus `json:"status"`
	StatusMtime int64         `json:"status_mtime"`
	Metadata    string        `json:"metadata"`
}

func aggregatedMetadataCheckFromChild(child *JugglerChild) AggregatedMetadataCheck {
	return AggregatedMetadataCheck{
		child.InstanceName,
		child.Actual.Status,
		child.Actual.StatusMtime,
		child.Actual.Metadata,
	}
}

type AggregatedMetadata struct {
	Timestamp       int64                     `json:"timestamp"`
	Reason          string                    `json:"reason"`
	PassedChecks    []AggregatedMetadataCheck `json:"passed_checks"`
	FailedChecks    []AggregatedMetadataCheck `json:"failed_checks"`
	MissedChecks    []AggregatedMetadataCheck `json:"missed_checks"`
	SuspectedChecks []AggregatedMetadataCheck `json:"suspected_checks"`
}

func stripChecks(children []JugglerChild) []AggregatedMetadataCheck {
	result := make([]AggregatedMetadataCheck, 0, len(children))
	for _, child := range children {
		result = append(result, aggregatedMetadataCheckFromChild(&child))
	}
	return result
}

func AggregateInfinibandPortChecks(children []JugglerChild, logger echo.Logger) []JugglerChild {
	oracleResultSize := len(children) / TypicalInfinibandPortsCount
	groupedChecks := make(map[string]*hostChecks, oracleResultSize)
	var oneHostChecks *hostChecks
	var ok bool
	for _, child := range children {
		oneHostChecks, ok = groupedChecks[child.HostName]
		if !ok {
			oneHostChecks = new(hostChecks)
			groupedChecks[child.HostName] = oneHostChecks
		}
		if child.IsMissing() {
			oneHostChecks.MissedChecks = append(oneHostChecks.MissedChecks, child)
		} else {
			switch walleStatus := child.Status.ToWalleStatus(); walleStatus {
			case WalleCheckStatusSuspected:
				oneHostChecks.SuspectedChecks = append(oneHostChecks.SuspectedChecks, child)
			case WalleCheckStatusFailed:
				oneHostChecks.FailedChecks = append(oneHostChecks.FailedChecks, child)
			case WalleCheckStatusPassed:
				oneHostChecks.PassedChecks = append(oneHostChecks.PassedChecks, child)
			default:
				logger.Errorf("Unknown walle status: %s", walleStatus)
			}
		}
	}
	res := make([]JugglerChild, 0, oracleResultSize)
	for _, oneHostChecks := range groupedChecks {
		res = append(res, oneHostChecks.GetAggregatedCheck())
	}
	return res
}
