package juggler

import (
	"encoding/json"
	"fmt"
	"time"
)

const (
	WalleRackFlappingTimeout = 20 * 60
	rackMinimalHostsBorder   = 5
)

type RackCheckType struct {
	WalleName            CheckType
	RackFailedPercents   int
	RackFlappingPercents int
	RackMissingPercents  int
	SuspectedPeriod      int64
}

type NewSmallRackMetadata struct {
	Total         int `json:"total"`
	MinimalAmount int `json:"minimal_amount"`
}

type NewNormalRackMetadata struct {
	Total             int   `json:"total"`
	Suspected         int   `json:"suspected"`
	Failed            int   `json:"failed"`
	ThresholdFlapping int   `json:"threshold_flapping"`
	ThresholdFailed   int   `json:"threshold_failed"`
	LastCrit          int64 `json:"last_crit"`
	SuspectedTimeout  int   `json:"suspected_timeout"`
}

var RackCheckMapping = map[CheckType]RackCheckType{
	// These percents may not play well for racks when too few hosts remaining.
	// But still it is better then just 50% => dead.
	CheckTypeUnreachable: {
		WalleName:            CheckTypeRack,
		RackFailedPercents:   80,
		RackFlappingPercents: 50,
		RackMissingPercents:  80,
		SuspectedPeriod:      10 * 60,
	},
	// Never flap never missing
	CheckTypeCPUCaches: {
		WalleName:            CheckTypeRackOverheat,
		RackFailedPercents:   50,
		RackFlappingPercents: 0,
		RackMissingPercents:  0,
		SuspectedPeriod:      5 * 60,
	},
}

type RackStatus struct {
	Type             RackCheckType
	TotalChilren     int
	JugglerStatus    JugglerStatus
	ReshapedChildren []JugglerChild
}

func (rackStatus *RackStatus) failedPercent(failedChildren int) int {
	return int(float64(failedChildren) / float64(rackStatus.TotalChilren) * 100)
}

func (rackStatus *RackStatus) missingPercent(missingChildren int) int {
	return int(float64(missingChildren) / float64(rackStatus.TotalChilren) * 100)
}

func (rackStatus *RackStatus) suspectedPercent(suspectedChildren int) int {
	return int(float64(suspectedChildren) / float64(rackStatus.TotalChilren) * 100)
}

func (rackStatus *RackStatus) TooSmall() bool {
	return rackStatus.TotalChilren < rackMinimalHostsBorder
}

func (rackStatus *RackStatus) failed(failedChildren int) bool {
	return rackStatus.failedPercent(failedChildren) >= rackStatus.Type.RackFailedPercents
}

func (rackStatus *RackStatus) flapping(failedChildren int, lastCritTimestamp int64) bool {
	isFlappingEnabled := rackStatus.Type.RackFlappingPercents > 0
	flappingTimeout := time.Now().Unix()-lastCritTimestamp <= WalleRackFlappingTimeout
	return isFlappingEnabled && rackStatus.failedPercent(failedChildren) >= rackStatus.Type.RackFlappingPercents && flappingTimeout
}

func (rackStatus *RackStatus) missing(missingChildren int) bool {
	isMissingEnabled := rackStatus.Type.RackMissingPercents > 0
	return isMissingEnabled && rackStatus.missingPercent(missingChildren) >= rackStatus.Type.RackMissingPercents
}

func (rackStatus *RackStatus) suspected(suspectedChildren int) bool {
	return rackStatus.suspectedPercent(suspectedChildren) >= rackStatus.Type.RackFailedPercents
}

func (rackStatus *RackStatus) StatusAndFlags(
	missingChildren int,
	suspectedChildren int,
	failedChildren int,
	lastCritTimestamp int64,
) (JugglerStatus, []string, []byte) {
	if rackStatus.TooSmall() {
		newSmallMetadata, err := json.Marshal(NewSmallRackMetadata{rackStatus.TotalChilren, rackMinimalHostsBorder})
		if err != nil {
			panic(fmt.Sprintf("get status and flags: %v", err))
		}
		return JugglerOk, jugglerCheckFiltersActualForceOk[:], newSmallMetadata
	}

	newMetadata, err := json.Marshal(NewNormalRackMetadata{
		rackStatus.TotalChilren,
		suspectedChildren,
		failedChildren,
		rackStatus.Type.RackFlappingPercents,
		rackStatus.Type.RackFailedPercents,
		lastCritTimestamp,
		WalleRackFlappingTimeout,
	})
	if err != nil {
		panic(fmt.Sprintf("get status and flags: %v", err))
	}

	//Here is the trick:
	//If we've got A% <= x < B% hosts failing then rack is suspected
	//If we've got B% <= x hosts failing then rack is dead
	//But if we've got A% <= x < B% hosts failing for too long then rack is probably alive. Suspected hosts
	//are considered in that case, if there are a lot of them, then something may be wrong anyway.
	if rackStatus.failed(failedChildren) {
		return JugglerCrit, jugglerCheckFiltersActual[:], newMetadata
	}
	if rackStatus.flapping(failedChildren, lastCritTimestamp) {
		return JugglerWarn, jugglerCheckFiltersActualFlapping[:], newMetadata
	}
	if rackStatus.missing(missingChildren) {
		return JugglerInfo, jugglerCheckFiltersActualMissing[:], newMetadata
	}
	if rackStatus.suspected(suspectedChildren) {
		return JugglerWarn, jugglerCheckFiltersActual[:], newMetadata
	}
	return JugglerOk, jugglerCheckFiltersActual[:], newMetadata
}

func GetRackStatus(checks []JugglerChild, rackType RackCheckType) RackStatus {
	var (
		missingChildren   int
		suspectedChildren int
		failedChildren    int
		lastCritTimestamp int64
		res               = RackStatus{Type: rackType, TotalChilren: len(checks)}
	)
	for _, child := range checks {
		var suspected bool
		if child.Actual.Status == JugglerCrit {
			if rackType.WalleName != CheckTypeRackOverheat || cpuOverheat(&child) {
				if time.Now().Unix()-child.Actual.StatusMtime > rackType.SuspectedPeriod {
					failedChildren++
					lastCritTimestamp = max(child.Actual.StatusMtime, lastCritTimestamp)
				} else {
					suspected = true
				}
			}
		}
		if child.Actual.Status == JugglerWarn || child.Actual.Status == JugglerInfo {
			suspected = true
		}
		if suspected {
			suspectedChildren++
		} else if child.IsMissing() {
			missingChildren++
		}
	}
	newJugglerStatus, rackFlags, newMetadata := res.StatusAndFlags(
		missingChildren, suspectedChildren, failedChildren, lastCritTimestamp,
	)
	for _, child := range checks {
		child.ServiceName = rackType.WalleName
		child.Status = newJugglerStatus
		child.Flags = rackFlags
		child.Actual.Status = newJugglerStatus
		child.Actual.Metadata = string(newMetadata)
		res.ReshapedChildren = append(res.ReshapedChildren, child)
	}
	return res
}

var CPUOverheatingMetadataResultMark = "CPU_OVERHEATING"

type OverheatMetadata struct {
	Result []string
}

func cpuOverheat(child *JugglerChild) bool {
	// TODO parse metadata before
	var metadata OverheatMetadata
	err := json.Unmarshal([]byte(child.Actual.Metadata), &metadata)
	if err == nil {
		return len(metadata.Result) == 1 && metadata.Result[0] == CPUOverheatingMetadataResultMark
	}
	return false
}

func max(x, y int64) int64 {
	if x < y {
		return y
	}
	return x
}

type RackTopologyStore interface {
	Get(string) map[string]int
}

func splitHalfrack(store RackTopologyStore, rackName string, checks []JugglerChild) [][]JugglerChild {
	var res [][]JugglerChild
	groupedHosts := make(map[int][]JugglerChild)
	rackTopology := store.Get(rackName)
	if rackTopology == nil {
		res = append(res, checks)
	} else {
		for _, child := range checks {
			rackRange := rackTopology[child.HostName]
			groupedHosts[rackRange] = append(groupedHosts[rackRange], child)
		}
		for _, groupedChildren := range groupedHosts {
			res = append(res, groupedChildren)
		}
	}
	return res
}
