package juggler

import (
	"encoding/json"
	"strings"
	"time"
)

type HostCheck struct {
	ID                 WalleCheckKey `bson:"_id"`
	FQDN               string        `bson:"fqdn"`
	Type               CheckType     `bson:"type"`
	Status             WalleStatus   `bson:"status"`
	Metadata           string        `bson:"metadata"`
	StatusMtime        int64         `bson:"status_mtime,truncate"`
	Timestamp          int64         `bson:"timestamp,truncate"`
	EffectiveTimestamp int64         `bson:"effective_timestamp,truncate"`
}

func (c *HostCheck) Key() WalleCheckKey {
	return WalleCheckKey(MkCheckKey(c.FQDN, c.Type))
}

func ToWalleFormat(check JugglerChild, prev *HostCheck) *HostCheck {
	var status WalleStatus
	var metadata string

	validMetadata := validateMetadata(&check)
	if validMetadata {
		metadata = check.Actual.Metadata
	} else {
		metadata = ""
	}

	if !validMetadata {
		status = WalleCheckStatusInvalid
	} else if check.ContainFlag(CheckFilterNodata) {
		status = WalleCheckStatusMissing
	} else if check.ContainFlag(checkFilterFlapping) {
		status = WalleCheckStatusSuspected
	} else if check.ContainFlag(checkFilterInvalid) {
		status = WalleCheckStatusInvalid
	} else {
		status = check.Actual.Status.ToWalleStatus()
	}

	now := time.Now().Unix()
	statusMtime := check.Actual.StatusMtime
	if statusMtime == 0 {
		statusMtime = now
	}
	if check.IsWalleMetaChecks() || check.IsMissing() {
		if prev != nil {

			if prev.StatusMtime > 0 && prev.StatusMtime < statusMtime && prev.Status == status {
				statusMtime = prev.StatusMtime
			}
		}
	}
	// NOTE(rocco66): ignore metadata parsing until WALLE-3884
	effectiveTimestamp, _ := calculateEffectiveTimestamp(check, status, statusMtime, now)
	return &HostCheck{
		ID:                 check.Key(),
		FQDN:               check.HostName,
		Type:               check.ServiceName,
		Status:             status,
		Metadata:           metadata,
		StatusMtime:        statusMtime,
		Timestamp:          now,
		EffectiveTimestamp: effectiveTimestamp,
	}
}

func validateMetadata(check *JugglerChild) bool {
	metadata := check.Actual.Metadata
	if !isJSON(metadata) && (strings.Contains(metadata, "timed out after") || strings.Contains(metadata, "exited with code")) {
		return false
	}
	return true
}

func isJSON(str string) bool {
	var js json.RawMessage
	return json.Unmarshal([]byte(str), &js) == nil
}

func ConvertChecks(
	children []JugglerChild,
	oldChecks map[WalleCheckKey]*HostCheck,
) []*HostCheck {

	var res = make([]*HostCheck, 0, len(children))
	for _, jugglerCheck := range children {
		prevState, ok := oldChecks[jugglerCheck.Key()]
		if ok && jugglerCheck.AnyChanges(prevState.Status, prevState.StatusMtime, prevState.Metadata) {
			Metrics.ChangedEvents.Update(1)
		}
		res = append(res, ToWalleFormat(jugglerCheck, prevState))

	}
	return res
}
