package lib

import (
	"fmt"
	"syscall"

	extzap "go.uber.org/zap"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/log/zap/logrotate"
)

func init() {
	err := logrotate.RegisterLogrotateSink(syscall.SIGHUP)
	if err != nil {
		panic(err)
	}
}

type LoggerConfig struct {
	Level  string   `yaml:"level"`
	Format string   `yaml:"format"`
	Paths  []string `yaml:"paths"`
}

func newConfig(format string, level log.Level) (extzap.Config, error) {
	switch format {
	case "tskv":
		return zap.TSKVConfig(level), nil
	case "console":
		return zap.ConsoleConfig(level), nil
	case "cli":
		return zap.CLIConfig(level), nil
	case "json":
		return zap.JSONConfig(level), nil
	case "kv":
		return zap.KVConfig(level), nil
	default:
		return extzap.Config{}, fmt.Errorf("unknown config format: [%s]", format)
	}
}

func newLogger(config LoggerConfig) (*zap.Logger, error) {
	level, err := log.ParseLevel(config.Level)
	if err != nil {
		return nil, err
	}
	zapConfig, err := newConfig(config.Format, level)
	if err != nil {
		return nil, err
	}
	zapConfig.OutputPaths = config.Paths

	logger, err := zap.New(zapConfig)
	if err != nil {
		return nil, err
	}
	return logger, nil
}

func NewLoggers(config map[string]LoggerConfig) (map[string]*zap.Logger, error) {
	loggers := make(map[string]*zap.Logger, len(config))

	for name, params := range config {
		logger, err := newLogger(params)
		if err != nil {
			return nil, err
		}
		loggers[name] = logger.WithName(name).(*zap.Logger)
	}
	return loggers, nil
}
