package monitoring

import (
	"errors"
	"math"
	"reflect"
	"time"

	"a.yandex-team.ru/library/go/yandex/unistat"
)

func getMetricSliceFromStruct(v interface{}) ([]unistat.Metric, error) {
	value := reflect.ValueOf(v)
	if value.Kind() != reflect.Struct {
		return nil, errors.New("not struct")
	}
	metrics := make([]unistat.Metric, value.NumField())
	for i := range metrics {
		var ok bool
		if metrics[i], ok = value.Field(i).Interface().(unistat.Metric); !ok {
			return nil, errors.New("not metric field")
		}
	}
	return metrics, nil
}

func TimeBuckets() []float64 {
	return append(append(newSlice(0, 0.1, 0.01), newSlice(0.2, 1, 0.1)...), newSlice(2, 10, 1)...)
}

func TimeBucketsLog() []float64 {
	size := 30
	buckets := make([]float64, size)
	buckets[0] = math.Pow(1.5, -4)
	for i := 1; i < size; i++ {
		buckets[i] = buckets[i-1] * 1.5
	}
	return buckets
}

func newSlice(start, end, step float64) []float64 {
	if step <= 0 || end < start {
		return []float64{}
	}
	s := make([]float64, 0, int(1+(end-start)/step))
	for start <= end {
		s = append(s, start)
		start += step
	}
	return s
}

func MeasureSecondsSince(m unistat.Metric, ts time.Time) {
	if m, ok := m.(unistat.Updater); ok {
		dur := time.Since(ts)
		m.Update(float64(dur) / float64(time.Second))
	}
}

func AnnnSuffix() unistat.StructuredAggregation {
	return unistat.StructuredAggregation{
		AggregationType: unistat.Absolute,
		Group:           unistat.Min,
		MetaGroup:       unistat.Min,
		Rollup:          unistat.Min,
	}
}
