package monitoring

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"os"
	"sync"
	"time"

	"github.com/labstack/echo/v4"
)

type JugglerStatus int

const (
	JugglerStatusOk JugglerStatus = iota
	JugglerStatusWarn
	JugglerStatusCrit
)

const JugglerEventServicePrefix = "wall-e."

type JugglerEvent struct {
	Service     string
	Status      JugglerStatus
	Description string
	Tags        []string
}

type JugglerAgentConfig struct {
	Enable       bool     `yaml:"enable"`
	Source       string   `yaml:"source"`
	Address      string   `yaml:"address"`
	Host         string   `yaml:"host,omitempty"`
	Instance     string   `yaml:"instance,omitempty"`
	Tags         []string `yaml:"tags,omitempty"`
	Installation string   `yaml:"installation"`
	Env          string   `yaml:"env"`
}

type JugglerAgentClient interface {
	Push(event *JugglerEvent) error
	Stop()
}

type jugglerAgentIdleClient struct{}

type jugglerAgentHTTPClient struct {
	client   *http.Client
	source   string
	address  string
	template jugglerFullEvent
	stopped  bool
	host     string
	mu       sync.RWMutex
}

type jugglerFullEvent struct {
	Host        string   `json:"host"`
	Service     string   `json:"service"`
	Instance    string   `json:"instance,omitempty"`
	Status      string   `json:"status"`
	Description string   `json:"description"`
	Tags        []string `json:"tags,omitempty"`
}

type jugglerRequest struct {
	Source string              `json:"source"`
	Events []*jugglerFullEvent `json:"events"`
}

type jugglerResponse struct {
	AcceptedEvents int  `json:"accepted_events"`
	Success        bool `json:"success"`
}

func (s JugglerStatus) String() string {
	return [...]string{"OK", "WARN", "CRIT"}[s]
}

func NewJugglerAgentClient(config JugglerAgentConfig) (JugglerAgentClient, error) {
	if !config.Enable {
		return &jugglerAgentIdleClient{}, nil
	}
	host := config.Host
	if host == "" {
		var err error
		if host, err = os.Hostname(); err != nil {
			return nil, err
		}
	}
	template := jugglerFullEvent{
		Host:     JugglerEventHost(config.Env, config.Installation),
		Service:  JugglerEventServicePrefix,
		Instance: config.Instance,
		Tags:     config.Tags,
	}
	return &jugglerAgentHTTPClient{
		client: &http.Client{
			Timeout: 5 * time.Second,
		},
		source:   config.Source,
		address:  config.Address,
		template: template,
		host:     host,
	}, nil
}

func (ev *JugglerEvent) WithTags(tags ...string) *JugglerEvent {
	ev.Tags = append(ev.Tags, tags...)
	return ev
}

func (jugglerAgentIdleClient) Push(*JugglerEvent) error {
	return nil
}

func (jugglerAgentIdleClient) Stop() {}

func (c *jugglerAgentHTTPClient) Push(event *JugglerEvent) error {
	ev := c.template
	ev.Service += event.Service
	ev.Status = event.Status.String()
	ev.Description = event.Description
	if event.Status == JugglerStatusCrit {
		ev.Description = appendHostInfo(ev.Description, c.host)
	}
	ev.Tags = append(ev.Tags, event.Tags...)

	return c.send(&ev)
}

func (c *jugglerAgentHTTPClient) Stop() {
	c.mu.Lock()
	defer c.mu.Unlock()
	c.stopped = true
	c.client.CloseIdleConnections()
}

func (c *jugglerAgentHTTPClient) send(event *jugglerFullEvent) error {
	c.mu.RLock()
	defer c.mu.RUnlock()

	if c.stopped {
		return errors.New("client is stopped")
	}

	req := jugglerRequest{
		Source: c.source,
		Events: []*jugglerFullEvent{event},
	}

	data, err := json.Marshal(&req)
	if err != nil {
		return err
	}

	httpReq, err := http.NewRequest("POST", c.address, bytes.NewBuffer(data))
	if err != nil {
		return err
	}
	httpReq.Header.Set(echo.HeaderContentType, echo.MIMEApplicationJSON)

	resp, err := c.client.Do(httpReq)
	if err != nil {
		return err
	}

	defer func() {
		_ = resp.Body.Close()
	}()

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("bad HTTP code from juggler agent: [%d]", resp.StatusCode)
	}

	var response jugglerResponse
	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		return err
	}

	if !response.Success || response.AcceptedEvents < 1 {
		return errors.New("event not accepted")
	}

	return nil
}

func appendHostInfo(desc, host string) string {
	return desc + " Host: " + host
}

func JugglerEventHost(env, installation string) string {
	return fmt.Sprintf("wall-e.srv.go.%s.%s", env, installation)
}
