package monitoring

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/suite"
)

type JugglerAgentHTTPClientSuite struct {
	suite.Suite
}

func (suite *JugglerAgentHTTPClientSuite) TestPush() {
	const (
		source       = "wall-e.test"
		host         = "host"
		installation = "installation"
		env          = "env"
	)
	desc := func(i int) string {
		return fmt.Sprintf("desc %d", i)
	}
	cases := []struct {
		statusCode JugglerStatus
		status     string
		tags       []string
	}{
		{
			statusCode: JugglerStatusOk,
			status:     "OK",
		},
		{
			statusCode: JugglerStatusCrit,
			status:     "CRIT",
			tags:       []string{"tag1"},
		},
		{
			statusCode: JugglerStatusWarn,
			status:     "WARN",
			tags:       []string{"tag1", "tag2"},
		},
	}
	type jugglerEvent struct {
		Host        string   `json:"host"`
		Service     string   `json:"service"`
		Instance    string   `json:"instance,omitempty"`
		Status      string   `json:"status"`
		Description string   `json:"description"`
		Digest      string   `json:"digest"`
		Tags        []string `json:"tags,omitempty"`
		OpenTime    float64  `json:"open_time"`
		StatusMtime float64  `json:"status_mtime"`
		Heartbeat   int      `json:"heartbeat"`
	}
	type jugglerRequest struct {
		Source string          `json:"source"`
		Events []*jugglerEvent `json:"events"`
	}
	type jugglerResponseEvent struct {
		Code  int    `json:"code"`
		Error string `json:"error"`
	}
	type jugglerResponse struct {
		Events         []*jugglerResponseEvent `json:"events"`
		AcceptedEvents int                     `json:"accepted_events"`
		Message        string                  `json:"message"`
		Success        bool                    `json:"success"`
	}
	caseNumber := 0
	jugglerTestHandle := func(w http.ResponseWriter, r *http.Request) {
		defer func() { caseNumber++ }()
		if r.Method != http.MethodPost {
			http.Error(w, "405 Method not allowed", http.StatusMethodNotAllowed)
			return
		}
		if r.Header.Get("content-type") != "application/json" {
			http.Error(w, "400 Bad request", http.StatusBadRequest)
			return
		}
		request := &jugglerRequest{}
		if err := json.NewDecoder(r.Body).Decode(request); err != nil {
			http.Error(w, "400 Bad request", http.StatusBadRequest)
			return
		}
		suite.Equal(source, request.Source)
		suite.NotEqual(0, len(request.Events))

		for _, e := range request.Events {
			suite.Equal(JugglerEventHost(env, installation), e.Host)
			suite.Equal(fmt.Sprintf("%s%d", JugglerEventServicePrefix, caseNumber), e.Service)
			suite.Equal(cases[caseNumber].status, e.Status)
			description := desc(caseNumber)
			if e.Status == JugglerStatusCrit.String() {
				description = appendHostInfo(description, host)
			}
			suite.Equal(description, e.Description)
			suite.Equal(cases[caseNumber].tags, e.Tags)
		}

		responseEvents := make([]*jugglerResponseEvent, 0)
		for range request.Events {
			responseEvents = append(responseEvents, &jugglerResponseEvent{Code: 200})
		}
		response := &jugglerResponse{
			Events:         responseEvents,
			AcceptedEvents: len(responseEvents),
			Success:        true,
		}
		w.Header().Set("Content-Type", "application/json")
		w.WriteHeader(http.StatusOK)
		_ = json.NewEncoder(w).Encode(response)
	}
	jugglerTestServer := httptest.NewServer(http.HandlerFunc(jugglerTestHandle))
	defer jugglerTestServer.Close()
	cnf := JugglerAgentConfig{
		Enable:       true,
		Source:       source,
		Address:      jugglerTestServer.URL,
		Host:         host,
		Tags:         make([]string, 0),
		Env:          env,
		Installation: installation,
	}
	client, err := NewJugglerAgentClient(cnf)
	suite.Require().NoError(err)
	for i, c := range cases {
		event := &JugglerEvent{
			Service:     fmt.Sprintf("%d", i),
			Status:      c.statusCode,
			Description: desc(i),
			Tags:        c.tags,
		}
		suite.NoError(client.Push(event))
	}

}

func TestJugglerAgentHTTPClient(t *testing.T) {
	suite.Run(t, new(JugglerAgentHTTPClientSuite))
}
