package netutil

import (
	"regexp"
	"strings"
)

type IP string

type MAC string

type SwitchName string

type SwitchPort string

var domainRegexp = regexp.MustCompile(`(?i)^(?:[a-z0-9]+(?:-[a-z0-9]+)*\.)+[a-z]{2,63}$`)

var switchDomains = []string{".yndx.net", ".netinfra.cloud.yandex.net"}

func IsValidFQDN(fqdn string) bool {
	return len(fqdn) < 253 && domainRegexp.MatchString(fqdn)
}

func ShortenRacktablesSwitchName(name SwitchName) SwitchName {
	nameStr := strings.ToLower(string(name))
	for _, domain := range switchDomains {
		nameStr = strings.TrimSuffix(nameStr, domain)
	}
	return SwitchName(nameStr)
}

func ShortenRacktablesPortName(port SwitchPort) SwitchPort {
	name := string(port)
	if r := regexp.MustCompile(`(?i)^eth-trunk(\d+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `Eth-Trunk${1}`)))
	}
	if r := regexp.MustCompile(`^(?:[Ee]thernet|Eth)\s*(.+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `e${1}`)))
	}
	if r := regexp.MustCompile(`^FastEthernet\s*(.+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `fa${1}`)))
	}
	if r := regexp.MustCompile(`^GigabitEthernet\s*(.+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `gi${1}`)))
	}
	if r := regexp.MustCompile(`^TenGigabitEthernet\s*(.+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `te${1}`)))
	}
	if r := regexp.MustCompile(`(?i)^port-channel\s*(.+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `po${1}`)))
	}
	if r := regexp.MustCompile(`^(?:XGigabitEthernet|XGE)\s*(.+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `xg${1}`)))
	}
	if r := regexp.MustCompile(`^LongReachEthernet\s*(.+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `lo${1}`)))
	}
	if r := regexp.MustCompile(`^ManagementEthernet\s*(.+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `ma${1}`)))
	}
	if r := regexp.MustCompile(`^Et\s*(\d.*)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `e${1}`)))
	}
	if r := regexp.MustCompile(`^TenGigE\s*(.*)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `te${1}`)))
	}
	if r := regexp.MustCompile(`^Mg(?:mtEth)?(.*)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `mg${1}`)))
	}
	if r := regexp.MustCompile(`^BE(\d+)$`); r.MatchString(name) {
		return SwitchPort(strings.ToLower(r.ReplaceAllString(name, `bundle-ether${1}`)))
	}
	return SwitchPort(strings.ToLower(name))
}

func IsSliceIPEqual(a, b []IP) bool {
	if len(a) == len(b) {
		for index, elemA := range a {
			if elemA != b[index] {
				return false
			}
		}
		return true
	}
	return false
}
