package tool

import (
	"sync"
	"time"
)

type CacheStore[K comparable, V any] struct {
	updater         func() (map[K]V, error)
	ttl             time.Duration
	dataLocker      sync.RWMutex
	timestamp       time.Time
	data            map[K]V
	operationLocker sync.Mutex
}

func (c *CacheStore[K, V]) Init(updater func() (map[K]V, error), ttl time.Duration) {
	c.data = make(map[K]V, 0)
	c.updater = updater
	c.ttl = ttl
}

func (c *CacheStore[K, V]) Get(key K) (V, error) {
	elapsed := time.Since(c.getTimestamp())
	var err error
	switch {
	case elapsed > c.ttl:
		err = c.update()
	case elapsed > c.ttl/2:
		go func() {
			_ = c.update()
		}()
	}
	c.dataLocker.RLock()
	defer c.dataLocker.RUnlock()
	return c.data[key], err
}

func (c *CacheStore[K, V]) update() error {
	c.operationLocker.Lock()
	defer c.operationLocker.Unlock()
	if time.Since(c.getTimestamp()) < c.ttl/2 {
		return nil
	}
	data, err := c.updater()
	if err == nil {
		c.dataLocker.Lock()
		c.timestamp = time.Now()
		c.data = data
		c.dataLocker.Unlock()
	}
	return err
}

func (c *CacheStore[K, V]) getTimestamp() time.Time {
	c.dataLocker.RLock()
	defer c.dataLocker.RUnlock()
	return c.timestamp
}
