package metrics

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"sort"
	"strings"
	"time"

	monitoring "a.yandex-team.ru/infra/monitoring/go/pkg/metrics"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
)

const (
	formatYASM    = "yasm"
	formatSolomon = "solomon"
)

var errInvalidFormats = errors.New("invalid 'formats' field")

type regFactories map[string]func(prefix string) monitoring.Registry

func newRegFactories(rawFormats interface{}) (regFactories, error) {
	known := map[string]func(string) monitoring.Registry{
		formatYASM: func(string) monitoring.Registry {
			return monitoring.NewUnistatRegistry(
				monitoring.UnistatRegistryOptions{TagStringifier: tagStringifier{}, PrefixSeparator: "-"},
			)
		},
		formatSolomon: func(prefix string) monitoring.Registry {
			return monitoring.NewSolomonRegistry().WithPrefix(prefix)
		},
	}
	res := make(map[string]func(string) monitoring.Registry)
	if rawFormats, ok := rawFormats.([]interface{}); ok {
		for _, format := range rawFormats {
			format, _ := format.(string)
			if known[format] == nil {
				return nil, errInvalidFormats
			}
			res[format] = known[format]
		}
	}
	if len(res) == 0 {
		return nil, errInvalidFormats
	}
	return res, nil
}

func (f regFactories) newRegistries(prefix string) *registries {
	var regs []monitoring.Registry
	composition := make(map[string]monitoring.Registry)
	for format, factory := range f {
		r := factory(prefix)
		regs = append(regs, r)
		composition[format] = r
	}
	return &registries{united: monitoring.NewUnitedRegistry(regs), composition: composition}
}

type registries struct {
	united      monitoring.Registry
	composition map[string]monitoring.Registry
}

func (r *registries) newMetricGroup(id string, timestamp int64) (*repos.MetricGroup, error) {
	metricGroup := &repos.MetricGroup{ID: id, Time: timestamp}
	bins := map[string][]byte{formatYASM: []byte("[]"), formatSolomon: []byte("{}")}
	var err error
	for format := range bins {
		if r.composition[format] != nil {
			bins[format], err = r.composition[format].Serialize()
			if err != nil {
				return nil, err
			}
		}
	}
	if err = json.Unmarshal(bins[formatYASM], &metricGroup.YasmData); err != nil {
		return nil, err
	}

	if err = json.Unmarshal(bins[formatSolomon], &metricGroup.SolomonData); err != nil {
		return nil, err
	}
	return metricGroup, nil
}

type tagStringifier struct{}

func (tagStringifier) ToString(tags map[string]string) string {
	keys := make([]string, 0, len(tags))
	for key := range tags {
		keys = append(keys, key)
	}
	sort.Strings(keys)
	result := ""
	for _, key := range keys {
		result += fmt.Sprintf("%s-%s-", key, tags[key])
	}
	return result
}

func (job *collectMetricsJob) saveMetrics(ctx context.Context, collectedData *data, timestamp time.Time) error {
	collectedData.computeExpertiseAgeQuantiles()
	if err := job.saveHostMetrics(ctx, collectedData, timestamp); err != nil {
		return err
	}
	if err := job.saveTaskMetrics(ctx, collectedData, timestamp); err != nil {
		return err
	}
	return job.saveHealthMetrics(ctx, collectedData, timestamp)
}

func (job *collectMetricsJob) saveHostMetrics(ctx context.Context, collectedData *data, timestamp time.Time) error {

	registries := job.formats.newRegistries("hosts")
	registry := registries.united
	// host number metrics
	writeNumericMetric(registry.WithPrefix("total"), metricNameTotal(), collectedData.Hosts.Num)
	for project, total := range collectedData.Hosts.ByProjects {
		writeNumericMetric(registry.WithTags(projectTags(project)), metricNameTotal(), total)
	}
	for tier, total := range collectedData.Hosts.ByTiers {
		writeNumericMetric(registry.WithTags(tierTags(tier)), metricNameTotal(), total)
	}

	// host state metrics
	for _, state := range allHostStates {
		writeNumericMetric(registry.WithPrefix("total"), string(state), collectedData.Hosts.States.TotalDistribution[state])
		for project, distr := range collectedData.Hosts.States.DistributionByProjects {
			writeNumericMetric(registry.WithTags(projectTags(project)), string(state), distr[state])
		}
		for tier, distr := range collectedData.Hosts.States.DistributionByTiers {
			writeNumericMetric(registry.WithTags(tierTags(tier)), string(state), distr[state])
		}
	}

	// expertize age metrics
	for level, quantile := range collectedData.ExpertiseAgeQuantiles.total {
		writeNumericMetric(registry.WithPrefix("total"), metricNameExpertiseAge(level), int(quantile))
	}
	for project, distr := range collectedData.ExpertiseAgeQuantiles.byProjects {
		for level, quantile := range distr {
			writeNumericMetric(registry.WithTags(projectTags(project)), metricNameExpertiseAge(level), int(quantile))
		}
	}
	for tier, distr := range collectedData.ExpertiseAgeQuantiles.byTiers {
		for level, quantile := range distr {
			writeNumericMetric(registry.WithTags(tierTags(tier)), metricNameExpertiseAge(level), int(quantile))
		}
	}

	// host decision metrics
	for decision, total := range collectedData.Hosts.Decisions.TotalDistribution {
		writeNumericMetric(registry.WithPrefix("total-decision"), decision, total)
	}
	subReg := registry.WithPrefix("decision")
	for project, distr := range collectedData.Hosts.Decisions.DistributionByProjects {
		for decision, total := range distr {
			writeNumericMetric(subReg.WithTags(projectTags(project)), decision, total)
		}
	}
	for tier, distr := range collectedData.Hosts.Decisions.DistributionByTiers {
		for decision, total := range distr {
			writeNumericMetric(subReg.WithTags(tierTags(tier)), decision, total)
		}
	}

	// host failure metrics
	for check, distr := range collectedData.Hosts.Failures["total"].Distribution {
		for status, total := range distr {
			writeNumericMetric(registry.WithPrefix("total"), metricNameFailure(check, status), total)
		}
	}

	for check, distr := range collectedData.Hosts.Failures["rtc"].Distribution {
		for status, total := range distr {
			writeNumericMetric(registry.WithPrefix("rtc"), metricNameFailure(check, status), total)
		}
	}

	// health check status metrics
	subReg = registry.WithPrefix("total-health-check")
	notZeroStatuses := make(map[string]struct{})
	for _, item := range collectedData.HealthChecks.Statuses {
		writeNumericMetric(subReg, metricNameCheckStatus(item.Type, item.Status), item.Total)
		notZeroStatuses[fmt.Sprintf("%s|%s", item.Type, item.Status)] = struct{}{}
	}
	for _, t := range juggler.AllCheckTypes() {
		for _, s := range juggler.AllWalleStatuses() {
			if _, ok := notZeroStatuses[fmt.Sprintf("%s|%s", t, s)]; !ok {
				writeNumericMetric(subReg, metricNameCheckStatus(t, s), 0)
			}
		}
	}

	// error report metrics
	writeNumericMetric(registry.WithPrefix("total"), metricNameErrorReportHost(), collectedData.ErrorReportHosts.Total)
	for project, total := range collectedData.ErrorReportHosts.DistributionByProjects {
		writeNumericMetric(registry.WithTags(projectTags(project)), metricNameErrorReportHost(), total)
	}

	// agent error metrics
	writeNumericMetric(registry.WithPrefix("total"), metricNameAgentErrors(), collectedData.Hosts.AgentErrors.Total)
	for project, total := range collectedData.Hosts.AgentErrors.DistributionByProjects {
		writeNumericMetric(registry.WithTags(projectTags(project)), metricNameAgentErrors(), total)
	}
	for tier, total := range collectedData.Hosts.AgentErrors.DistributionByTiers {
		writeNumericMetric(registry.WithTags(tierTags(tier)), metricNameAgentErrors(), total)
	}

	// calculated metrics
	reachable := collectedData.Hosts.States.TotalDistribution[hostStateHealthy] +
		collectedData.Hosts.Failures["total"].Reachable
	writeNumericMetric(registry, metricNameReachable(), reachable)

	metricGroup, err := registries.newMetricGroup("hosts", timestamp.Unix())
	if err != nil {
		return err
	}
	return job.metrics.Update(ctx, metricGroup)
}

func (job *collectMetricsJob) saveHealthMetrics(
	ctx context.Context,
	collectedData *data,
	timestamp time.Time,
) error {
	registries := job.formats.newRegistries("health")
	reg := registries.united
	writeHistogramMetric(reg, collectedData.HealthChecks.Ages, "timestamp")
	writeHistogramMetric(reg, collectedData.HealthChecks.AgesEffective, "effective_timestamp")
	metricGroup, err := registries.newMetricGroup("health", timestamp.Unix())
	if err != nil {
		return err
	}
	return job.metrics.Update(ctx, metricGroup)
}

func (job *collectMetricsJob) saveTaskMetrics(ctx context.Context, collectedData *data, timestamp time.Time) error {
	registries := job.formats.newRegistries("tasks")
	registry := registries.united

	for taskType, taskData := range collectedData.Hosts.Tasks {
		for metric, total := range taskData.TotalDistribution {
			writeNumericMetric(registry.WithPrefix("total"), metricNameTask(taskType, metric), total)
		}
		for project, distr := range taskData.DistributionByProjects {
			for metric, total := range distr {
				writeNumericMetric(registry.WithTags(projectTags(project)), metricNameTask(taskType, metric), total)
			}
		}
		for tier, distr := range taskData.DistributionByTiers {
			for metric, total := range distr {
				writeNumericMetric(registry.WithTags(tierTags(tier)), metricNameTask(taskType, metric), total)
			}
		}
	}
	for logType, data := range collectedData.AuditLogs {
		writeNumericMetric(registry.WithPrefix("total"), logType, data.Total)
		for project, total := range data.DistributionByProjects {
			writeNumericMetric(registry.WithTags(projectTags(project)), logType, total)
		}
	}
	metricGroup, err := registries.newMetricGroup("tasks", timestamp.Unix())
	if err != nil {
		return err
	}
	return job.metrics.Update(ctx, metricGroup)
}

func metricNameCheckStatus(t juggler.CheckType, s juggler.WalleStatus) string {
	return fmt.Sprintf("%s-%s", strings.ReplaceAll(string(t), "_", "-"), s)
}

func metricNameTotal() string {
	return "total"
}

func metricNameHealthCheckAge(ageField string, t juggler.CheckType) string {
	return fmt.Sprintf("health_%s_age_%s", ageField, t)
}

func metricNameExpertiseAge(level float64) string {
	return fmt.Sprintf("expertize-age-%d", int(level*100))
}

func metricNameTask(taskType, metric string) string {
	return fmt.Sprintf("%s-%s", taskType, metric)
}

func metricNameFailure(check string, status juggler.WalleStatus) string {
	check = strings.ReplaceAll(check, "_", "-")
	return fmt.Sprintf("%s-%s", check, status)
}

func metricNameErrorReportHost() string {
	return "hosts-in-reports"
}

func metricNameAgentErrors() string {
	return "walle-agent-errors"
}

func metricNameReachable() string {
	return "total-reachable"
}

func newNumeric(name string) *monitoring.Numeric {
	return monitoring.NewNumeric(
		name,
		monitoring.StructuredAggregation{
			Group:     monitoring.LastAR,
			MetaGroup: monitoring.LastAR,
			Rollup:    monitoring.LastAR,
		},
		monitoring.LastAR,
	)
}

func tierTags(tier int) map[string]string {
	return map[string]string{"tier": fmt.Sprintf("%d", tier)}
}

func projectTags(project repos.ProjectID) map[string]string {
	return map[string]string{"walle-project": string(project)}
}

func writeNumericMetric(registry monitoring.Registry, name string, number int) {
	m := newNumeric(name)
	registry.RegisterNumeric(m)
	m.Update(float64(number))
}

func writeHistogramMetric(reg monitoring.Registry, aggregations []*repos.HealthCheckAgeAggregation, ageField string) {
	ages := make(map[juggler.CheckType]map[float64]int64)
	for _, t := range juggler.AllCheckTypes() {
		ages[t] = make(map[float64]int64)
	}
	for _, item := range aggregations {
		ages[item.Type][item.Age] = item.Total
	}

	for checkType, totals := range ages {
		histogram := monitoring.NewHistogram(
			metricNameHealthCheckAge(ageField, checkType),
			monitoring.AbsoluteHT,
			ageBuckets(),
		)
		reg.RegisterHistogram(histogram)
		var bucketValues []int64
		for _, b := range ageBuckets() {
			bucketValues = append(bucketValues, totals[b])
		}
		histogram.Init(bucketValues)

	}
}
