package metrics

import (
	"context"
	"errors"
	"sort"
	"strconv"
	"strings"
	"time"

	"github.com/gofrs/uuid"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/automation"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/cron"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	"a.yandex-team.ru/infra/walle/server/go/internal/utilities"
	lockrepo "a.yandex-team.ru/infra/walle/server/go/internal/utilities/repository"
	"a.yandex-team.ru/library/go/core/log"
)

const lockID = "cron/collect-metrics"

// hostState is a metric computed on base of host's state, host's status, project's automation
type hostState string

const (
	hostStateMaintenance   = hostState(automation.HostStateMaintenance)
	hostStateProbation     = hostState(automation.HostStateProbation)
	hostStateDead          = hostState(automation.HostStatusDead)
	hostStateInvalid       = hostState(automation.HostStatusInvalid)
	hostStateAutomated     = hostState("automated")
	hostStateHealthy       = hostState("healthy")
	hostStateHealthMissing = hostState("no-health")
)

const auditLogActualizationPeriod = 5 * time.Minute

var allHostStates = []hostState{
	hostStateMaintenance,
	hostStateProbation,
	hostStateDead,
	hostStateInvalid,
	hostStateAutomated,
	hostStateHealthy,
	hostStateHealthMissing,
}

type collectMetricsJob struct {
	locker          utilities.Locker
	health          HealthRepo
	hosts           HostRepo
	projects        ProjectRepo
	metrics         MetricRepo
	shards          ShardRepo
	errorReports    ErrorReportRepo
	auditLogs       AuditLogRepo
	automationPlots AutomationPlotRepo
	logger          log.Logger
	formats         regFactories
	expertSystem    *lib.ExpertSystemConfig
}

func NewCollectMetricsJob(config *cron.JobConfig) (cron.Job, error) {
	formats, err := newRegFactories(config.Extra["formats"])
	if err != nil {
		return nil, err
	}

	lockRepo := lockrepo.NewLockRepository(config.Store.YDB)
	locker := utilities.NewLocker(lockRepo)
	return &collectMetricsJob{
		locker:          locker,
		health:          repos.NewHealthRepo(config.Store.MongoDBHealth, config.Store.MongoReadPrefHealth),
		hosts:           repos.NewHostRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		projects:        repos.NewProjectRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		metrics:         repos.NewMetricRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		shards:          repos.NewShardRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		errorReports:    repos.NewErrorReportRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		auditLogs:       repos.NewAuditLogRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		automationPlots: repos.NewAutomationPlotRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		logger:          config.Logger,
		formats:         formats,
		expertSystem:    config.GlobalConfig.ExpertSystem,
	}, nil
}

func (job *collectMetricsJob) Do(ctx context.Context) (executed bool, err error) {
	lockOwner := uuid.Must(uuid.NewV4())
	lock, err := job.locker.Lock(job.logger, lockID, lockOwner.String())
	if err != nil {
		return executed, err
	}
	if lock == nil {
		return executed, nil
	}
	defer job.locker.UnlockAt(lock, cron.NextTimeFromContext(ctx))
	executed = true

	collectedData := &data{}
	timestamp := time.Now()
	g, gctx := errgroup.WithContext(ctx)
	g.Go(func() (err error) {
		collectedData.ShardExpertises, err = job.loadShardData(gctx)
		if err != nil {
			return
		}
		collectedData.Hosts, err = job.loadHostData(gctx, collectedData.ShardExpertises)
		return
	})
	g.Go(func() (err error) {
		collectedData.HealthChecks, err = job.loadHealthCheckData(gctx)
		return
	})
	g.Go(func() (err error) {
		collectedData.ErrorReportHosts, err = job.loadErrorReportData(gctx)
		return
	})
	g.Go(func() (err error) {
		collectedData.AuditLogs, err = job.loadAuditLogs(gctx, timestamp)
		return
	})
	if err = g.Wait(); err != nil {
		return executed, err
	}
	return executed, job.saveMetrics(ctx, collectedData, timestamp)
}

func (job *collectMetricsJob) loadHostData(ctx context.Context, shardExpertises []*shardExpertise) (*hosts, error) {
	automatedProjects, err := job.getAutomatedProjects(ctx)
	if err != nil {
		return nil, err
	}
	rtcProjects, err := job.getRTCProjects(ctx)
	if err != nil {
		return nil, err
	}
	automationPlotChecks, err := job.getAutomationPlotChecks(ctx)
	if err != nil {
		return nil, err
	}
	selection, err := job.hosts.Select(
		ctx,
		&repos.HostFilter{},
		[]string{
			repos.HostFieldKeyState,
			repos.HostFieldKeyStatus,
			repos.HostFieldKeyProject,
			repos.HostFieldKeyTier,
			repos.HostFieldKeyRestriction,
			repos.HostFieldKeyHealth,
			repos.HostFieldKeyInv,
			repos.HostFieldKeyTask,
			repos.HostFieldKeyAgentErrorFlag,
		},
	)
	if err != nil {
		return nil, err
	}
	hostData := newHostData(automationPlotChecks)
	for selection.Next() {
		host := &automation.Host{}
		var agentErrorFlag bool
		if err = selection.Scan(
			&host.State,
			&host.Status,
			&host.ProjectID,
			&host.Tier,
			&host.Restrictions,
			&host.Health,
			&host.Inv,
			&host.Task,
			&agentErrorFlag,
		); err != nil {
			return nil, err
		}
		hostData.Num++
		hostData.ByProjects[host.ProjectID]++
		hostData.ByTiers[host.Tier]++
		hostData.loadStates(host, automatedProjects)
		hostData.loadShardData(host, job.expertSystem)
		hostData.loadDecisionData(host)
		hostData.loadTaskData(host)
		// failure data
		if !host.IsHealthy() && host.Health != nil {
			hostData.Failures["total"].addData(host.Health.Reasons)
			if _, ok := rtcProjects[host.ProjectID]; ok {
				hostData.Failures["rtc"].addData(host.Health.Reasons)
			}
		}
		// agent errors data
		if agentErrorFlag {
			hostData.AgentErrors.Total++
			hostData.AgentErrors.DistributionByProjects[host.ProjectID]++
			hostData.AgentErrors.DistributionByTiers[host.Tier]++
		}
	}
	return hostData, nil
}

func (job *collectMetricsJob) getAutomatedProjects(ctx context.Context) (map[repos.ProjectID]struct{}, error) {
	list, err := job.projects.FindFieldValues(ctx, &repos.ProjectFilter{Automated: true}, repos.ProjectFieldKeyID)
	if err != nil {
		return nil, err
	}
	result := make(map[repos.ProjectID]struct{})
	for _, id := range list {
		result[repos.ProjectID(id)] = struct{}{}
	}
	return result, nil
}

func (job *collectMetricsJob) loadShardData(ctx context.Context) ([]*shardExpertise, error) {
	list, err := job.shards.Find(ctx, &repos.ShardFilter{Type: repos.ShardTypeScreening})
	if err != nil {
		return nil, err
	}
	if len(list) == 0 {
		return nil, errors.New("empty shard list")
	}
	now := time.Now().Unix()
	results := make([]*shardExpertise, len(list))
	for i, item := range list {
		tier, err := strconv.Atoi(item.Tier)
		if err != nil {
			return nil, err
		}
		results[i] = &shardExpertise{shard: &shard{id: item.ID, tier: tier}, age: now - item.ProcessingTime}
	}
	sort.Slice(results, func(i, j int) bool { return results[i].age < results[j].age })
	return results, nil
}

func (job *collectMetricsJob) getRTCProjects(ctx context.Context) (map[repos.ProjectID]struct{}, error) {
	list, err := job.projects.FindFieldValues(ctx, &repos.ProjectFilter{RTC: true}, repos.ProjectFieldKeyID)
	if err != nil {
		return nil, err
	}
	result := make(map[repos.ProjectID]struct{})
	for _, id := range list {
		result[repos.ProjectID(id)] = struct{}{}
	}
	return result, nil
}

func (job *collectMetricsJob) loadHealthCheckData(ctx context.Context) (*repos.HealthMetricAggregation, error) {
	return job.health.AggregateMetrics(ctx, ageBuckets())
}

func (job *collectMetricsJob) loadErrorReportData(ctx context.Context) (*errorReportHosts, error) {
	hostsInReports := &errorReportHosts{
		DistributionByProjects: make(map[repos.ProjectID]int),
	}
	selection, err := job.errorReports.Select(
		ctx,
		&repos.ErrorReportFilter{Open: true},
		[]string{repos.ErrorReportFieldKeyHosts},
	)
	if err != nil {
		return nil, err
	}
	for selection.Next() {
		var hosts []*repos.ErrorReportHost
		if err = selection.Scan(&hosts); err != nil {
			return nil, err
		}
		for _, h := range hosts {
			if !h.Solved {
				hostsInReports.Total++
				hostsInReports.DistributionByProjects[h.Project]++
			}
		}
	}
	return hostsInReports, nil
}

func (job *collectMetricsJob) loadAuditLogs(ctx context.Context, timestamp time.Time) (map[string]*auditLogs, error) {
	logs := map[string]*auditLogs{
		"dns-fixes": {
			DistributionByProjects: make(map[repos.ProjectID]int),
		},
		"admin-requests": {
			DistributionByProjects: make(map[repos.ProjectID]int),
		},
	}
	selection, err := job.auditLogs.Select(
		ctx,
		&repos.AuditLogFilter{Time: timestamp.Add(-auditLogActualizationPeriod)},
		[]string{repos.AuditLogFieldKeyType, repos.AuditLogFieldKeyProject},
	)
	if err != nil {
		return nil, err
	}
	for selection.Next() {
		var logType, projectStr string
		if err = selection.Scan(&logType, &projectStr); err != nil {
			return nil, err
		}
		if projectStr != "" {
			project := repos.ProjectID(projectStr)
			switch {
			case logType == automation.AuditLogTypeFixDNSRecords:
				logs["dns-fixes"].Total++
				logs["dns-fixes"].DistributionByProjects[project]++
			case strings.HasPrefix(logType, automation.AdminRequestPrefix):
				logs["admin-requests"].Total++
				logs["admin-requests"].DistributionByProjects[project]++
			}
		}
	}
	return logs, nil
}

func (job *collectMetricsJob) getAutomationPlotChecks(ctx context.Context) ([]string, error) {
	return job.automationPlots.ListAllChecks(ctx)
}

type hosts struct {
	Num         int
	ByProjects  map[repos.ProjectID]int
	ByTiers     map[int]int
	States      *hostStates
	Shards      *hostShards
	Decisions   *hostDecisions
	Tasks       map[string]*hostTasks
	Failures    map[string]*hostFailures
	AgentErrors *hostAgentErrors
}

func newHostData(automationPlotChecks []string) *hosts {
	return &hosts{
		ByProjects: make(map[repos.ProjectID]int),
		ByTiers:    make(map[int]int),
		States: &hostStates{
			DistributionByProjects: make(map[repos.ProjectID]map[hostState]int),
			DistributionByTiers:    make(map[int]map[hostState]int),
			TotalDistribution:      make(map[hostState]int),
		},
		Shards: &hostShards{
			DistributionByProjects: make(map[repos.ProjectID]map[shard]int),
			NumberByProjects:       make(map[repos.ProjectID]int),
			DistributionByTiers:    make(map[int]map[shard]int),
			NumberByTiers:          make(map[int]int),
			TotalDistribution:      make(map[shard]int),
		},
		Decisions: &hostDecisions{
			DistributionByProjects: make(map[repos.ProjectID]map[string]int),
			DistributionByTiers:    make(map[int]map[string]int),
			TotalDistribution:      make(map[string]int),
		},
		Tasks: map[string]*hostTasks{
			"user": {
				DistributionByProjects: make(map[repos.ProjectID]map[string]int),
				DistributionByTiers:    make(map[int]map[string]int),
				TotalDistribution:      newTaskDistribution(),
			},
			"auto": {
				DistributionByProjects: make(map[repos.ProjectID]map[string]int),
				DistributionByTiers:    make(map[int]map[string]int),
				TotalDistribution:      newTaskDistribution(),
			},
		},
		Failures: map[string]*hostFailures{
			"total": {
				Distribution: newFailureDistribution(automationPlotChecks),
			},
			"rtc": {
				Distribution: newFailureDistribution(automationPlotChecks),
			},
		},
		AgentErrors: &hostAgentErrors{
			DistributionByProjects: make(map[repos.ProjectID]int),
			DistributionByTiers:    make(map[int]int),
		},
	}
}

func (hostData *hosts) loadStates(host *automation.Host, automatedProjects map[repos.ProjectID]struct{}) {
	if _, ok := hostData.States.DistributionByProjects[host.ProjectID]; !ok {
		hostData.States.DistributionByProjects[host.ProjectID] = make(map[hostState]int)
	}
	if _, ok := hostData.States.DistributionByTiers[host.Tier]; !ok {
		hostData.States.DistributionByTiers[host.Tier] = make(map[hostState]int)
	}
	switch {
	case host.State == automation.HostStateProbation:
		hostData.States.TotalDistribution[hostStateProbation]++
		hostData.States.DistributionByProjects[host.ProjectID][hostStateProbation]++
		hostData.States.DistributionByTiers[host.Tier][hostStateProbation]++

	case host.State == automation.HostStateMaintenance:
		hostData.States.TotalDistribution[hostStateMaintenance]++
		hostData.States.DistributionByProjects[host.ProjectID][hostStateMaintenance]++
		hostData.States.DistributionByTiers[host.Tier][hostStateMaintenance]++
	}
	switch {
	case host.Status == automation.HostStatusDead:
		hostData.States.TotalDistribution[hostStateDead]++
		hostData.States.DistributionByProjects[host.ProjectID][hostStateDead]++
		hostData.States.DistributionByTiers[host.Tier][hostStateDead]++

	case host.Status == automation.HostStatusInvalid:
		hostData.States.TotalDistribution[hostStateInvalid]++
		hostData.States.DistributionByProjects[host.ProjectID][hostStateInvalid]++
		hostData.States.DistributionByTiers[host.Tier][hostStateInvalid]++
	}
	if _, ok := automatedProjects[host.ProjectID]; ok {
		if host.IsAutomated() {
			hostData.States.TotalDistribution[hostStateAutomated]++
			hostData.States.DistributionByProjects[host.ProjectID][hostStateAutomated]++
			hostData.States.DistributionByTiers[host.Tier][hostStateAutomated]++
		}
	}
	switch {
	case host.IsHealthy():
		hostData.States.TotalDistribution[hostStateHealthy]++
		hostData.States.DistributionByProjects[host.ProjectID][hostStateHealthy]++
		hostData.States.DistributionByTiers[host.Tier][hostStateHealthy]++
	case host.IsHealthMissing():
		hostData.States.TotalDistribution[hostStateHealthMissing]++
		hostData.States.DistributionByProjects[host.ProjectID][hostStateHealthMissing]++
		hostData.States.DistributionByTiers[host.Tier][hostStateHealthMissing]++
	}
}

func (hostData *hosts) loadShardData(host *automation.Host, expertSystem *lib.ExpertSystemConfig) {
	shardsNum := expertSystem.ShardsNum[host.Tier]
	if shardsNum == 0 {
		shardsNum = expertSystem.ShardsNumDefault
	}
	hostShard := shard{id: automation.GetShardID(host.Inv, shardsNum), tier: host.Tier}
	hostData.Shards.TotalDistribution[hostShard]++
	if _, ok := hostData.Shards.DistributionByProjects[host.ProjectID]; !ok {
		hostData.Shards.DistributionByProjects[host.ProjectID] = make(map[shard]int)
	}
	if _, ok := hostData.Shards.DistributionByTiers[host.Tier]; !ok {
		hostData.Shards.DistributionByTiers[host.Tier] = make(map[shard]int)
	}
	hostData.Shards.DistributionByProjects[host.ProjectID][hostShard]++
	hostData.Shards.NumberByProjects[host.ProjectID]++
	hostData.Shards.DistributionByTiers[host.Tier][hostShard]++
	hostData.Shards.NumberByTiers[host.Tier]++
}

func (hostData *hosts) loadDecisionData(host *automation.Host) {
	if host.Health != nil && host.Health.Decision != nil && host.Health.Decision.Action != "" {
		hostData.Decisions.TotalDistribution[host.Health.Decision.Action]++
		if _, ok := hostData.Decisions.DistributionByProjects[host.ProjectID]; !ok {
			hostData.Decisions.DistributionByProjects[host.ProjectID] = make(map[string]int)
		}
		if _, ok := hostData.Decisions.DistributionByTiers[host.Tier]; !ok {
			hostData.Decisions.DistributionByTiers[host.Tier] = make(map[string]int)
		}
		hostData.Decisions.DistributionByProjects[host.ProjectID][host.Health.Decision.Action]++
		hostData.Decisions.DistributionByTiers[host.Tier][host.Health.Decision.Action]++
	}
}

func (hostData *hosts) loadTaskData(host *automation.Host) {
	if operation, ok := automation.Operations[automation.HostOperationStatus(host.Status)]; ok {
		var taskData *hostTasks
		if host.Task.Type == automation.TaskTypeManual {
			taskData = hostData.Tasks["user"]
		} else {
			taskData = hostData.Tasks["auto"]
		}
		metric := operation.Metric
		if metric == "" {
			metric = "other"
		}
		taskData.TotalDistribution[metric]++
		if _, ok := taskData.DistributionByProjects[host.ProjectID]; !ok {
			taskData.DistributionByProjects[host.ProjectID] = make(map[string]int)
		}
		if _, ok := taskData.DistributionByTiers[host.Tier]; !ok {
			taskData.DistributionByTiers[host.Tier] = make(map[string]int)
		}
		taskData.DistributionByProjects[host.ProjectID][metric]++
		taskData.DistributionByTiers[host.Tier][metric]++
	}
}

type quantiles struct {
	total      map[float64]int64
	byProjects map[repos.ProjectID]map[float64]int64
	byTiers    map[int]map[float64]int64
}

type data struct {
	Hosts                 *hosts
	HealthChecks          *repos.HealthMetricAggregation
	ShardExpertises       []*shardExpertise
	ErrorReportHosts      *errorReportHosts
	AuditLogs             map[string]*auditLogs
	ExpertiseAgeQuantiles quantiles
}

func (d *data) computeExpertiseAgeQuantiles() {
	levels := []float64{0.01, 0.75, 0.99}

	d.ExpertiseAgeQuantiles = quantiles{
		total:      make(map[float64]int64, len(levels)),
		byProjects: make(map[repos.ProjectID]map[float64]int64, len(d.Hosts.Shards.NumberByProjects)),
		byTiers:    make(map[int]map[float64]int64, len(d.Hosts.Shards.NumberByTiers)),
	}
	percentileIndex := 0
	var portion float64
	projectPortions := make(map[repos.ProjectID]float64)
	projectPercentileIndexes := make(map[repos.ProjectID]int)
	tierPortions := make(map[int]float64)
	tierPercentileIndexes := make(map[int]int)
	for _, exp := range d.ShardExpertises {
		portion += float64(d.Hosts.Shards.TotalDistribution[*exp.shard]) / float64(d.Hosts.Num)
		for percentileIndex < len(levels) && portion > levels[percentileIndex] {
			d.ExpertiseAgeQuantiles.total[levels[percentileIndex]] = exp.age
			percentileIndex++
		}
		for project, distribution := range d.Hosts.Shards.DistributionByProjects {
			projectPortions[project] += float64(distribution[*exp.shard]) /
				float64(d.Hosts.Shards.NumberByProjects[project])
			for projectPercentileIndexes[project] < len(levels) && projectPortions[project] > levels[projectPercentileIndexes[project]] {
				if _, ok := d.ExpertiseAgeQuantiles.byProjects[project]; !ok {
					d.ExpertiseAgeQuantiles.byProjects[project] = make(map[float64]int64, len(levels))
				}
				d.ExpertiseAgeQuantiles.byProjects[project][levels[projectPercentileIndexes[project]]] = exp.age
				projectPercentileIndexes[project]++
			}
		}
		for tier, distribution := range d.Hosts.Shards.DistributionByTiers {
			tierPortions[tier] += float64(distribution[*exp.shard]) / float64(d.Hosts.Shards.NumberByTiers[tier])
			for tierPercentileIndexes[tier] < len(levels) && tierPortions[tier] > levels[tierPercentileIndexes[tier]] {
				if _, ok := d.ExpertiseAgeQuantiles.byTiers[tier]; !ok {
					d.ExpertiseAgeQuantiles.byTiers[tier] = make(map[float64]int64, len(levels))
				}
				d.ExpertiseAgeQuantiles.byTiers[tier][levels[tierPercentileIndexes[tier]]] = exp.age
				tierPercentileIndexes[tier]++
			}
		}
	}
}

type hostStates struct {
	DistributionByProjects map[repos.ProjectID]map[hostState]int
	DistributionByTiers    map[int]map[hostState]int
	TotalDistribution      map[hostState]int
}

type hostShards struct {
	DistributionByProjects map[repos.ProjectID]map[shard]int
	NumberByProjects       map[repos.ProjectID]int
	DistributionByTiers    map[int]map[shard]int
	NumberByTiers          map[int]int
	TotalDistribution      map[shard]int
}

type hostDecisions struct {
	DistributionByProjects map[repos.ProjectID]map[string]int
	DistributionByTiers    map[int]map[string]int
	TotalDistribution      map[string]int
}

type hostTasks struct {
	DistributionByProjects map[repos.ProjectID]map[string]int
	DistributionByTiers    map[int]map[string]int
	TotalDistribution      map[string]int
}

type hostFailures struct {
	Distribution map[string]map[juggler.WalleStatus]int
	Reachable    int
}

func (f *hostFailures) addData(reasons []string) {
	foundCheckGroups := make(map[string]bool)
	reachable := true
	for _, reason := range reasons {
		check, status := automation.SplitFailureReason(reason)
		if status == "" {
			continue
		}
		if _, ok := f.Distribution[check]; !ok {
			f.Distribution[check] = make(map[juggler.WalleStatus]int)
		}
		f.Distribution[check][status]++

		fullCheckType := healthCheckTypes[check]
		if fullCheckType == "" {
			fullCheckType = juggler.CheckType(check)
		}
		if juggler.IsActiveCheckType(fullCheckType) {
			key := healthCheckGroupActive + "-" + string(status)
			if !foundCheckGroups[key] {
				foundCheckGroups[key] = true
				f.Distribution[healthCheckGroupActive][status]++
				reachable = false
			}
		}
		if juggler.IsMetaCheckType(fullCheckType) {
			key := healthCheckGroupMeta + "-" + string(status)
			if !foundCheckGroups[key] {
				foundCheckGroups[key] = true
				f.Distribution[healthCheckGroupMeta][status]++
			}
		}
		if juggler.IsPassiveCheckType(fullCheckType) {
			key := healthCheckGroupPassive + "-" + string(status)
			if !foundCheckGroups[key] {
				foundCheckGroups[key] = true
				f.Distribution[healthCheckGroupPassive][status]++
			}
		}
		if juggler.IsWalleCheckType(fullCheckType) {
			key := healthCheckGroupWalle + "-" + string(status)
			if !foundCheckGroups[key] {
				foundCheckGroups[key] = true
				f.Distribution[healthCheckGroupWalle][status]++
			}
		}
		if juggler.IsNetmonCheckType(fullCheckType) {
			key := healthCheckGroupNetmon + "-" + string(status)
			if !foundCheckGroups[key] {
				foundCheckGroups[key] = true
				f.Distribution[healthCheckGroupNetmon][status]++
			}
		}
	}
	if reachable {
		f.Reachable++
	}
}

type hostAgentErrors struct {
	DistributionByProjects map[repos.ProjectID]int
	DistributionByTiers    map[int]int
	Total                  int
}

type errorReportHosts struct {
	DistributionByProjects map[repos.ProjectID]int
	Total                  int
}

type auditLogs struct {
	DistributionByProjects map[repos.ProjectID]int
	Total                  int
}

func ageBuckets() []float64 {
	return []float64{0, 60, 120, 300, 600, 1200, 3600, 7200, 18000, 86400, 864000, 8640000}
}

type shardExpertise struct {
	age   int64
	shard *shard
}

type shard struct {
	id   int
	tier int
}

var healthCheckTypes = map[string]juggler.CheckType{
	"unreachable":    juggler.CheckTypeUnreachable,
	"ssh":            juggler.CheckTypeSSH,
	"meta":           juggler.CheckTypeMeta,
	"memory":         juggler.CheckTypeMemory,
	"disk":           juggler.CheckTypeDisk,
	"link":           juggler.CheckTypeLink,
	"reboots":        juggler.CheckTypeReboots,
	"tainted_kernel": juggler.CheckTypeTaintedKernel,
	"cpu":            juggler.CheckTypeCPU,
	"cpu_caches":     juggler.CheckTypeCPUCaches,
	"cpu_capping":    juggler.CheckTypeCPUCapping,
	"fs_check":       juggler.CheckTypeFSCheck,
	"bmc":            juggler.CheckTypeBmc,
	"gpu":            juggler.CheckTypeGpu,
	"infiniband":     juggler.CheckTypeInfiniband,
	"rack":           juggler.CheckTypeRack,
	"rack_overheat":  juggler.CheckTypeRackOverheat,
	"switch":         juggler.CheckTypeNetmon,
	"tor_link":       juggler.CheckTypeTorLink,
	"ib_link":        juggler.CheckTypeIbLink,
}

const (
	healthCheckGroupActive  = "active-checks"
	healthCheckGroupMeta    = "meta-checks"
	healthCheckGroupPassive = "passive-checks"
	healthCheckGroupWalle   = "walle-checks"
	healthCheckGroupNetmon  = "netmon-checks"
)

func newTaskDistribution() map[string]int {
	result := make(map[string]int)
	result["other"] = 0
	for _, operation := range automation.Operations {
		if operation.Metric != "" {
			result[operation.Metric] = 0
		}
	}
	return result
}

func newFailureDistribution(automationPlotChecks []string) map[string]map[juggler.WalleStatus]int {
	result := make(map[string]map[juggler.WalleStatus]int)
	for check := range healthCheckTypes {
		result[check] = make(map[juggler.WalleStatus]int)
		for _, status := range []juggler.WalleStatus{
			juggler.WalleCheckStatusFailed,
			juggler.WalleCheckStatusSuspected,
		} {
			result[check][status] = 0
		}
	}
	for _, group := range []string{
		healthCheckGroupWalle,
		healthCheckGroupMeta,
		healthCheckGroupActive,
		healthCheckGroupPassive,
		healthCheckGroupNetmon,
	} {
		result[group] = make(map[juggler.WalleStatus]int)
		for _, status := range []juggler.WalleStatus{
			juggler.WalleCheckStatusFailed,
			juggler.WalleCheckStatusSuspected,
			juggler.WalleCheckStatusMissing,
		} {
			result[group][status] = 0
		}
	}
	for _, automationPlotCheck := range automationPlotChecks {
		result[automationPlotCheck] = make(map[juggler.WalleStatus]int)
		for _, status := range []juggler.WalleStatus{
			juggler.WalleCheckStatusFailed,
			juggler.WalleCheckStatusSuspected,
		} {
			result[automationPlotCheck][status] = 0
		}
	}
	return result
}
