package monitoring

import (
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"math"
	"os"
	"os/exec"
	"time"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib"
	libcron "a.yandex-team.ru/infra/walle/server/go/internal/lib/cron"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/monitoring"
)

const chatNotificationTag = "dev-chat-notifications"

const maxJugglerFlapVal = 86400

type jugglerCheckConfig struct {
	Namespace   string                     `json:"namespace,omitempty"`
	Host        string                     `json:"host,omitempty"`
	Service     string                     `json:"service,omitempty"`
	RefreshTime int                        `json:"refresh_time,omitempty"`
	TTL         int                        `json:"ttl,omitempty"`
	Aggregator  string                     `json:"aggregator,omitempty"`
	Kwargs      map[string]string          `json:"aggregator_kwargs,omitempty"`
	Flaps       *jugglerCheckConfigFlaps   `json:"flaps,omitempty"`
	Tags        []string                   `json:"tags,omitempty"`
	Children    []*jugglerCheckConfigChild `json:"children,omitempty"`
	Description string                     `json:"description,omitempty"`
}

type jugglerCheckConfigFlaps struct {
	Stable   int `json:"stable,omitempty"`
	Critical int `json:"critical,omitempty"`
}

type jugglerCheckConfigChild struct {
	Host     string `json:"host,omitempty"`
	Service  string `json:"service,omitempty"`
	Type     string `json:"group_type,omitempty"`
	Instance string `json:"instance,omitempty"`
}

type config struct {
	CronRegistry    libcron.RegistryConfig        `yaml:"cron"`
	JugglerAgent    monitoring.JugglerAgentConfig `yaml:"juggler_agent"`
	DeployProjectID string                        `yaml:"project_id"`
	DeployStageID   string                        `yaml:"deploy_stage_id"`
	DeployUnitID    string                        `yaml:"deploy_unit_id"`
}

func UpdateChecks(args *lib.CliArgs) {
	if err := os.Chdir(args.RootDir); err != nil {
		panic(err)
	}
	cnf := &config{}
	if err := lib.LoadConfig(args.ConfigPath, cnf); err != nil {
		panic(err)
	}
	if err := updateCronChecks(cnf, args.DryRun); err != nil {
		panic(err)
	}
}

func updateCronChecks(cnf *config, dryRun bool) error {
	if len(cnf.CronRegistry.Jobs) == 0 || !cnf.JugglerAgent.Enable {
		return nil
	}
	if os.Getenv("JUGGLER_OAUTH_TOKEN") == "" {
		return errors.New("not set environmental variable JUGGLER_OAUTH_TOKEN")
	}
	checks, err := newCronChecks(cnf)
	if err != nil {
		return err
	}
	mark := fmt.Sprintf("walle_go_cron_autogenerated_%s_%s", cnf.JugglerAgent.Env, cnf.JugglerAgent.Installation)
	sdkCmdArgs := []string{"load", "--mark", mark}
	if dryRun {
		sdkCmdArgs = append(sdkCmdArgs, "--dry-run")
	}
	cmd := exec.Command("juggler-sdk", sdkCmdArgs...)
	stdin, err := cmd.StdinPipe()
	if err != nil {
		return err
	}
	encoder := json.NewEncoder(stdin)
	errCh := make(chan error, 1)
	go func() {
		defer stdin.Close()
		errCh <- encoder.Encode(checks)
	}()

	out, err := cmd.CombinedOutput()
	defer log.Println("juggler-sdk run:\n", string(out))

	if err != nil {
		return err
	}
	return <-errCh
}

func newCronChecks(cnf *config) ([]*jugglerCheckConfig, error) {
	namespace := "wall-e.srv"
	host := monitoring.JugglerEventHost(cnf.JugglerAgent.Env, cnf.JugglerAgent.Installation)
	childAggregateService := "wall-e.cron.run_timeout"

	var checks []*jugglerCheckConfig
	var children []*jugglerCheckConfigChild
	var minHeartbeat int
	for jobName, jobConfig := range cnf.CronRegistry.Jobs {
		childAggregateHost := host + "." + jobName
		rawEventService := monitoring.JugglerEventServicePrefix + libcron.MonitoringServiceSuffix(jobName)

		jobInterval, _ := time.ParseDuration(jobConfig.Spec.Every)
		heartbeat := int(jobInterval.Seconds())
		if minHeartbeat == 0 || minHeartbeat > heartbeat {
			minHeartbeat = heartbeat
		}
		check := &jugglerCheckConfig{
			Namespace:  namespace,
			Host:       childAggregateHost,
			Service:    childAggregateService,
			TTL:        3 * heartbeat,
			Aggregator: "logic_or",
			Kwargs:     map[string]string{"nodata_mode": "force_crit"},
			Flaps: &jugglerCheckConfigFlaps{
				Stable:   int(math.Min(float64(3*heartbeat), maxJugglerFlapVal)),
				Critical: int(math.Min(float64(5*3*heartbeat), maxJugglerFlapVal)),
			},
			Children: []*jugglerCheckConfigChild{
				{
					Host:    host,
					Service: rawEventService,
					Type:    "HOST",
				},
			},
		}
		checks = append(checks, check)
		children = append(children, &jugglerCheckConfigChild{
			Host:     childAggregateHost,
			Service:  childAggregateService,
			Type:     "HOST",
			Instance: "all",
		})
	}
	checks = append(checks, &jugglerCheckConfig{
		Namespace:  namespace,
		Host:       host,
		Service:    "wall-e.cron",
		TTL:        3 * minHeartbeat,
		Aggregator: "logic_or",
		Kwargs:     map[string]string{"nodata_mode": "force_crit"},
		Children:   children,
		Tags:       []string{chatNotificationTag},
	})
	return checks, nil
}
