package repos

import (
	"context"
	"time"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
)

const auditLogCollectionName = "audit_log"

const (
	AuditLogFieldKeyID      = "_id"
	AuditLogFieldKeyTime    = "time"
	AuditLogFieldKeyType    = "type"
	AuditLogFieldKeyProject = "project"
)

type AuditLogStatus string

const AuditLogStatusCompleted AuditLogStatus = "completed"

type AuditLogType string

const (
	AuditLogTypeActiveMacChanged  AuditLogType = "active-mac-change-detected"
	AuditLogTypeIPsChanged        AuditLogType = "ips-change-detected"
	AuditLogTypeSwitchPortChanged AuditLogType = "switch-port-change-detected"
)

type AuditLog struct {
	ID         string                 `bson:"_id"`
	HostInv    int                    `bson:"host_inv"`
	HostName   string                 `bson:"host_name"`
	HostUUID   string                 `bson:"host_uuid"`
	Issuer     string                 `bson:"issuer"`
	Payload    map[string]interface{} `bson:"payload"`
	Project    string                 `bson:"project"`
	Status     AuditLogStatus         `bson:"status"`
	StatusTime float64                `bson:"status_time"`
	Time       float64                `bson:"time"`
	Type       AuditLogType           `bson:"type"`
}

type AuditLogRepo struct {
	collection *mongo.Collection
}

func NewAuditLogRepo(db *mongo.Database, pref *readpref.ReadPref) *AuditLogRepo {
	return &AuditLogRepo{
		collection: db.Collection(auditLogCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

func (repo *AuditLogRepo) Select(ctx context.Context, filter *AuditLogFilter, keys []string) (*db.MongoSelection, error) {
	filters := bson.D{}
	if filter != nil {
		filters = filter.getBSON()
	}
	return db.NewMongoSelection(ctx, repo.collection, filters, options.Find(), keys)
}

func (repo *AuditLogRepo) Insert(ctx context.Context, record *AuditLog) error {
	_, err := repo.collection.InsertOne(ctx, record)
	return err
}

type AuditLogFilter struct {
	Time time.Time
}

func (f *AuditLogFilter) getBSON() bson.D {
	filters := bson.D{}
	if !f.Time.IsZero() {
		filters = append(filters, bson.E{Key: AuditLogFieldKeyTime, Value: bson.M{"$gt": f.Time.Unix()}})
	}
	return filters
}
