package repos

import (
	"context"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"
)

const automationPlotCollectionName = "automation_plot"

const (
	AutomationPlotFieldKeyID     = "_id"
	AutomationPlotFieldKeyChecks = "checks"
)

type AutomationPlot struct {
	Checks []*AutomationPlotCheck `bson:"checks"`
}

type AutomationPlotCheck struct {
	Name string `bson:"name"`
}

type AutomationPlotRepo struct {
	collection *mongo.Collection
}

func NewAutomationPlotRepo(db *mongo.Database, pref *readpref.ReadPref) *AutomationPlotRepo {
	return &AutomationPlotRepo{
		collection: db.Collection(automationPlotCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

func (repo *AutomationPlotRepo) ListAllChecks(ctx context.Context) ([]string, error) {
	opts := (&options.FindOptions{}).SetProjection(bson.D{
		{Key: AutomationPlotFieldKeyChecks, Value: 1},
		{Key: AutomationPlotFieldKeyID, Value: 0},
	})
	cursor, err := repo.collection.Find(ctx, bson.D{}, opts)
	if err != nil {
		return nil, err
	}
	var plots []*AutomationPlot
	if err = cursor.All(ctx, &plots); err != nil {
		return nil, err
	}
	var result []string
	unique := make(map[string]struct{})
	for _, plot := range plots {
		for _, check := range plot.Checks {
			if _, ok := unique[check.Name]; !ok {
				unique[check.Name] = struct{}{}
				result = append(result, check.Name)
			}
		}
	}
	return result, nil
}
