package repos

import (
	"context"
	"fmt"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
)

const cacheCollectionName = "cache"

const (
	CacheFieldKeyID    = "_id"
	CacheFieldKeyValue = "value"
	CacheFieldKeyTime  = "time"
	RackFieldKeyName   = "aggregate"
	RackFieldKeyRanges = "hosts_ranges"
)

const standPrefix = "stand_hosts:"

type CacheItem struct {
	ID    string      `bson:"_id"`
	Value interface{} `bson:"value"`
	Time  int64       `bson:"time"`
}

type RackTopology struct {
	Name       string
	HostRanges map[string]int
}

type CacheRepo struct {
	collection *mongo.Collection
}

func NewCacheRepo(db *mongo.Database, pref *readpref.ReadPref) *CacheRepo {
	return &CacheRepo{
		collection: db.Collection(cacheCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

func (repo *CacheRepo) FindRackTopologies(ctx context.Context) (map[string]*RackTopology, error) {
	keys := []string{
		fmt.Sprintf("%s.%s", CacheFieldKeyValue, RackFieldKeyName),
		fmt.Sprintf("%s.%s", CacheFieldKeyValue, RackFieldKeyRanges),
	}
	res, err := repo.Select(ctx, &CacheFilter{ID: "^rack_topology:"}, keys)
	if err != nil {
		return nil, err
	}
	topology := make(map[string]*RackTopology)
	for res.Next() {
		rack := RackTopology{HostRanges: make(map[string]int)}
		if err = res.Scan(&rack.Name, &rack.HostRanges); err != nil {
			return nil, err
		}
		topology[rack.Name] = &rack
	}
	return topology, nil
}

func (repo *CacheRepo) GetStandHosts(ctx context.Context, stand string) ([]string, int64, error) {
	id := standPrefix + stand
	item := struct {
		Value []string `bson:"value"`
		Time  int64    `bson:"time"`
	}{}
	if err := repo.collection.FindOne(ctx, bson.M{CacheFieldKeyID: id}).Decode(&item); err != nil {
		return nil, 0, err
	}
	return item.Value, item.Time, nil
}

func (repo *CacheRepo) Select(ctx context.Context, filter *CacheFilter, keys []string) (*db.MongoSelection, error) {
	filters := bson.D{}
	if filter != nil {
		filters = filter.getBSON()
	}
	return db.NewMongoSelection(ctx, repo.collection, filters, options.Find(), keys)
}

func (repo *CacheRepo) Insert(ctx context.Context, item *CacheItem) error {
	_, err := repo.collection.InsertOne(ctx, item)
	return err
}

type CacheFilter struct {
	ID string
}

func (f *CacheFilter) getBSON() bson.D {
	filters := bson.D{}
	if f.ID != "" {
		filters = append(filters, bson.E{Key: CacheFieldKeyID, Value: bson.D{{Key: "$regex", Value: f.ID}}})
	}
	return filters
}
