package repos

import (
	"context"
	"fmt"
	"testing"
	"time"

	"github.com/stretchr/testify/suite"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
)

type CacheTestSuite struct {
	suite.Suite
	repo  *CacheRepo
	racks []*RackTopology
}

func (suite *CacheTestSuite) SetupSuite() {
	mongodb, err := db.GetTestingMongoDB()
	suite.Require().NoError(err)
	suite.repo = NewCacheRepo(mongodb, readpref.Primary())
}

func (suite *CacheTestSuite) TestFindRackTopologies() {
	var items []*CacheItem
	var racks []*RackTopology
	for i := 0; i < 2; i++ {
		rack := &RackTopology{Name: fmt.Sprintf("aggregate-%d", i), HostRanges: map[string]int{fmt.Sprintf("host-%d", i): i}}
		racks = append(racks, rack)
		items = append(items, &CacheItem{
			ID: fmt.Sprintf("rack_topology:%d", i),
			Value: map[string]interface{}{
				RackFieldKeyName:   rack.Name,
				RackFieldKeyRanges: rack.HostRanges,
				"foo":              "bar",
			},
		})
	}
	items = append(items, &CacheItem{
		ID:    "another",
		Value: map[string]interface{}{"foo": "bar"},
	})
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	for _, item := range items {
		suite.Require().NoError(suite.repo.Insert(ctx, item))
	}
	suite.racks = racks
	expected := map[string]*RackTopology{
		suite.racks[0].Name: suite.racks[0],
		suite.racks[1].Name: suite.racks[1],
	}

	topology, err := suite.repo.FindRackTopologies(ctx)
	suite.Require().NoError(err)
	suite.Equal(expected, topology)
}

func (suite *CacheTestSuite) TestGetStandHosts() {
	stand := "some_stand"
	standHosts := []string{"host1", "host2"}
	timestamp := time.Now().Unix()
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	_, err := suite.repo.collection.InsertOne(
		ctx,
		bson.M{"_id": standPrefix + stand, "value": standHosts, "time": timestamp},
	)
	suite.Require().NoError(err)
	gotHosts, gotTime, err := suite.repo.GetStandHosts(ctx, stand)
	suite.Require().NoError(err)
	suite.Assert().Equal(standHosts, gotHosts)
	suite.Assert().Equal(timestamp, gotTime)

}

func TestCacheRepo(t *testing.T) {
	suite.Run(t, new(CacheTestSuite))
}
