package repos

import (
	"context"
	"fmt"
	"strconv"
	"sync"
	"testing"
	"time"

	"github.com/stretchr/testify/suite"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler"
)

type HealthTestSuite struct {
	suite.Suite
	repo *HealthRepo
}

func (suite *HealthTestSuite) SetupSuite() {
	mongodb, err := db.GetTestingMongoDB()
	suite.Require().NoError(err)
	suite.repo = NewHealthRepo(mongodb, readpref.Primary())
}

func (suite *HealthTestSuite) TearDownTest() {
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	_, err := suite.repo.collection.DeleteMany(ctx, bson.D{})
	suite.Require().NoError(err)
}

func (suite *HealthTestSuite) TestBulkUpsertChecks() {
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	checksBeforeWriting := make(map[juggler.WalleCheckKey]juggler.HostCheck)
	for i := 0; i < 3; i++ {
		check := juggler.HostCheck{
			ID:        juggler.WalleCheckKey(fmt.Sprintf("%d", i)),
			FQDN:      fmt.Sprintf("host-%d", i),
			Status:    juggler.WalleStatus(fmt.Sprintf("status-%d", i)),
			Timestamp: int64(i),
		}
		_, err := suite.repo.collection.InsertOne(ctx, check)
		suite.Require().NoError(err)
		checksBeforeWriting[check.ID] = check
	}
	checksToWrite := make([]*juggler.HostCheck, 3)
	check0 := checksBeforeWriting["0"]
	checksToWrite[0] = &check0
	checksToWrite[0].Status = "status-new"
	checksToWrite[0].Timestamp++
	check1 := checksBeforeWriting["1"]
	checksToWrite[1] = &check1
	checksToWrite[1].Status = "status-new"
	checksToWrite[2] = &juggler.HostCheck{
		ID:     "new",
		FQDN:   "host",
		Status: "status",
	}
	_, err := suite.repo.BulkUpsert(ctx, checksToWrite)

	suite.Require().NoError(err)
	var res []juggler.HostCheck
	cur, err := suite.repo.collection.Find(ctx, bson.D{{}})
	suite.Require().NoError(err)
	err = cur.All(ctx, &res)
	suite.Require().NoError(err)
	suite.Equal(4, len(res))
	expected := map[juggler.WalleCheckKey]juggler.HostCheck{
		"0":   *checksToWrite[0],
		"1":   checksBeforeWriting["1"],
		"2":   checksBeforeWriting["2"],
		"new": *checksToWrite[2],
	}
	for _, writtenCheck := range res {
		suite.Equal(expected[writtenCheck.ID], writtenCheck)
	}
}

func (suite *HealthTestSuite) TestHealthBulkWriter() {
	w, err := suite.repo.NewBulkWriter(&HealthBulkWriterOptions{
		Size:          7,
		FlushInterval: 500 * time.Millisecond,
		RPSLimit:      50,
	})
	suite.Require().NoError(err)
	go w.Run()
	checksBulks := make([][]*juggler.HostCheck, 10)
	for i := 0; i < 10; i++ {
		for j := 0; j < 10; j++ {
			checksBulks[i] = append(
				checksBulks[i],
				&juggler.HostCheck{ID: juggler.WalleCheckKey(fmt.Sprintf("%d", j)), Timestamp: int64(i * j)},
			)
		}
	}
	wg := sync.WaitGroup{}
	for _, bulk := range checksBulks {
		wg.Add(1)
		go func(checks []*juggler.HostCheck) {
			defer wg.Done()
			w.Add(checks)
		}(bulk)
	}
	wg.Wait()
	w.Shutdown()
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	cur, err := suite.repo.collection.Find(ctx, bson.D{{}})
	suite.Require().NoError(err)
	var res []juggler.HostCheck

	suite.Require().NoError(cur.All(ctx, &res))
	suite.Require().Equal(10, len(res))
	for _, check := range res {
		id, _ := strconv.Atoi(string(check.ID))
		suite.Equal(id*9, int(check.Timestamp))
	}
}

func (suite *HealthTestSuite) TestHealthBulkWriterRateLimit() {
	ticks := 0
	opts := &HealthBulkWriterOptions{
		Size:          10,
		FlushInterval: 500 * time.Millisecond,
		RPSLimit:      50,
		Handler: func(result *mongo.BulkWriteResult, err error) {
			ticks++
		},
	}
	w, err := suite.repo.NewBulkWriter(opts)
	suite.Require().NoError(err)
	m, n := 10, 10
	checksBulks := make([][]*juggler.HostCheck, m)
	for i := 0; i < m; i++ {
		for j := 0; j < n; j++ {
			checksBulks[i] = append(
				checksBulks[i],
				&juggler.HostCheck{ID: juggler.WalleCheckKey(fmt.Sprintf("%d", j)), Timestamp: int64(i * j)},
			)
		}
	}
	start := time.Now()
	go w.Run()
	for _, bulk := range checksBulks {
		w.Add(bulk)
	}
	w.Shutdown()
	end := time.Now()

	tickslimit := m * n / opts.Size
	suite.Require().GreaterOrEqual(ticks, tickslimit)

	timeLimit := time.Duration(float64(tickslimit)*1000/opts.RPSLimit) * time.Millisecond
	elapsed := end.Sub(start)
	suite.Require().GreaterOrEqual(elapsed.Microseconds(), timeLimit.Microseconds())

}

func TestHealthRepo(t *testing.T) {
	suite.Run(t, new(HealthTestSuite))
}
