package repos

import (
	"context"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	netutil "a.yandex-team.ru/infra/walle/server/go/internal/lib/net"
)

const hostCollectionName = "hosts"

const (
	HostFieldKeyID              = "_id"
	HostFieldKeyInv             = "inv"
	HostFieldKeyName            = "name"
	HostFieldKeyProject         = "project"
	HostFieldKeyState           = "state"
	HostFieldKeyStatus          = "status"
	HostFieldKeyTier            = "tier"
	HostFieldKeyRestriction     = "restriction"
	HostFieldKeyHealth          = "health"
	HostFieldKeyTask            = "task"
	HostFieldKeyAgentErrorFlag  = "walle_agent_errors_flag"
	HostFieldKeyLocation        = "location"
	HostFieldKeyMACs            = "macs"
	HostFieldKeyIPs             = "ips"
	HostFieldKeyActiveMac       = "active_mac"
	HostFieldKeyActiveMacSource = "active_mac_source"
	HostFieldKeyAgentVersion    = "agent_version"
	HostFieldKeyMessages        = "messages"

	HostLocationFieldKeySwitch        = "switch"
	HostLocationFieldKeyPort          = "port"
	HostLocationFieldKeyNetworkSource = "network_source"
)

type HostUUID string
type HostInv int
type HostName string
type WalleAgentVersion string

type Host struct {
	UUID         HostUUID          `bson:"_id"`
	Inv          HostInv           `bson:"inv"`
	Name         HostName          `bson:"name"`
	Project      ProjectID         `bson:"project"`
	State        string            `bson:"state"`
	Status       string            `bson:"status"`
	Location     *HostLocation     `bson:"location"`
	AgentVersion WalleAgentVersion `bson:"agent_version"`
}

type HostCommon struct {
	Name   HostName
	State  string
	Status string
}

// HostLocation is network, physical and logical location of a host
type HostLocation struct {
	Switch        netutil.SwitchName `bson:"switch"`
	Port          netutil.SwitchPort `bson:"port"`
	NetworkSource string             `bson:"network_source"`

	Country           string `bson:"country"`
	City              string `bson:"city"`
	Datacenter        string `bson:"datacenter"` // physical datacenter
	Queue             string `bson:"queue"`
	Rack              string `bson:"rack"`
	Unit              string `bson:"string"`
	PhysicalTimestamp int64  `bson:"physical_timestamp"` // Unix time of phys. location updating

	LogicalDatacenter string `bson:"logical_datacenter"`

	ShortDatacenterName string `bson:"short_datacenter_name"`
	ShortQueueName      string `bson:"short_queue_name"`
}

func NewHostRepo(db *mongo.Database, pref *readpref.ReadPref) *HostRepo {
	return &HostRepo{
		collection: db.Collection(hostCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

type HostRepo struct {
	collection *mongo.Collection
}

func (repo *HostRepo) Find(ctx context.Context, filter *HostFilter) ([]*Host, error) {
	opts := options.Find()
	var hosts []*Host
	cursor, err := repo.collection.Find(ctx, filter.getBSON(), opts)
	if err != nil {
		return nil, err
	}
	if err := cursor.All(ctx, &hosts); err != nil {
		return nil, err
	}
	return hosts, nil
}

func (repo *HostRepo) FindCommon(ctx context.Context, filter *HostFilter) ([]*HostCommon, error) {
	selection, err := repo.Select(ctx, filter, []string{HostFieldKeyName, HostFieldKeyState, HostFieldKeyStatus})
	if err != nil {
		return nil, err
	}
	var hosts []*HostCommon
	for selection.Next() {
		host := &HostCommon{}
		if err = selection.Scan(&host.Name, &host.State, &host.Status); err != nil {
			return nil, err
		}
		hosts = append(hosts, host)
	}
	return hosts, nil
}

func (repo *HostRepo) Select(ctx context.Context, filter *HostFilter, keys []string) (*db.MongoSelection, error) {
	return db.NewMongoSelection(ctx, repo.collection, filter.getBSON(), options.Find(), keys)
}

func (repo *HostRepo) SelectOne(ctx context.Context, name string, keys []string) (*db.MongoSelection, error) {
	return db.NewMongoSelection(
		ctx,
		repo.collection,
		bson.D{{Key: CacheFieldKeyID, Value: name}},
		options.Find().SetLimit(1),
		keys,
	)
}

func (repo *HostRepo) Insert(ctx context.Context, host *Host) error {
	_, err := repo.collection.InsertOne(ctx, host)
	return err
}

func (repo *HostRepo) NewBulkWriter(opts *db.MongoBulkWriterOptions) (db.BulkWriter, error) {
	return db.NewMongoBulkWriter(repo.collection, opts)
}

type HostFilter struct {
	Project ProjectID
	Tier    *HostFilterTier
	Shard   *HostFilterShard
}

type HostFilterTier struct {
	TierValue uint64
}

type HostFilterShard struct {
	ID  int
	Num int
}

func (f *HostFilter) getBSON() bson.D {
	filters := bson.D{}
	if f == nil {
		return filters
	}
	if f.Project != "" {
		filters = append(filters, bson.E{Key: HostFieldKeyProject, Value: f.Project})
	}
	if f.Tier != nil {
		filters = append(filters, bson.E{Key: HostFieldKeyTier, Value: f.Tier.TierValue})
	}
	if f.Shard != nil {
		filters = append(filters, bson.E{Key: HostFieldKeyInv, Value: bson.M{"$mod": [2]int{f.Shard.Num, f.Shard.ID}}})
	}
	return filters
}

type HostMessage struct {
	Severity string `bson:"severity"`
	Message  string `bson:"message"`
}

func HostMessageInfo(m string) HostMessage {
	return HostMessage{Severity: "info", Message: m}
}

func HostMessageError(m string) HostMessage {
	return HostMessage{Severity: "error", Message: m}
}
