package repos

import (
	"context"
	"time"

	"go.mongodb.org/mongo-driver/bson"

	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/juggler"
)

const hostDecisionCollectionName = "host_decision"

type DecisionAction string
type DecisionFailureType string
type DecisionReason string
type DecisionRestriction string
type DecisionFailure string

const (
	HostDecisionFieldKeyUUID          = "_id"
	HostDecisionFieldKeyDecision      = "decision"
	HostDecisionFieldKeyScreeningTime = "screening_time"
	HostDecisionFieldKeyCounter       = "counter"
)
const (
	DecisionActionHealthy = "healthy"
)

type RepairRuleName string

type Decision struct {
	Action       DecisionAction
	FailureType  DecisionFailureType `bson:"failure_type"`
	Reason       DecisionReason
	Params       map[string]string
	Restrictions []DecisionRestriction
	Checks       []juggler.CheckType
	Failures     []DecisionFailure
	RuleName     RepairRuleName `bson:"rule_name"`
}

func NewHostDecisionRepo(db *mongo.Database, pref *readpref.ReadPref) *HostDecisionRepo {
	return &HostDecisionRepo{
		collection: db.Collection(hostDecisionCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

type HostDecisionRepo struct {
	collection *mongo.Collection
}

func (repo *HostDecisionRepo) Upsert(ctx context.Context, uuid HostUUID, decision *Decision) error {
	opts := options.UpdateOptions{}
	opts.SetUpsert(true)
	filters := bson.D{
		bson.E{Key: HostDecisionFieldKeyUUID, Value: uuid},
	}
	update := bson.D{
		bson.E{Key: "$set", Value: bson.D{
			bson.E{Key: HostDecisionFieldKeyUUID, Value: uuid},
			bson.E{Key: HostDecisionFieldKeyDecision, Value: decision},
			bson.E{Key: HostDecisionFieldKeyScreeningTime, Value: time.Now().Unix()},
		}},
		bson.E{Key: "$inc", Value: bson.D{
			bson.E{Key: HostDecisionFieldKeyCounter, Value: 1},
		}},
	}
	_, err := repo.collection.UpdateOne(ctx, filters, update, &opts)
	return err
}
