package repos

import (
	"context"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	netutil "a.yandex-team.ru/infra/walle/server/go/internal/lib/net"
)

const hostNetworkCollectionName = "host_network"

const (
	HostNetworkFieldKeyID              = "_id"
	HostNetworkFieldKeyActiveMac       = "active_mac"
	HostNetworkFieldKeyActiveMacTime   = "active_mac_time"
	HostNetworkFieldKeyActiveMacSource = "active_mac_source"
	HostNetworkFieldKeyIPs             = "ips"
	HostNetworkFieldKeyIPsTime         = "ips_time"
	HostNetworkFieldKeySwitch          = "network_switch"
	HostNetworkFieldKeyPort            = "network_port"
	HostNetworkFieldKeyNetworkTime     = "network_timestamp"
	HostNetworkFieldKeyNetworkSource   = "network_source"
)

type HostNetwork struct {
	HostUUID        string             `bson:"_id"`
	ActiveMac       netutil.MAC        `bson:"active_mac"`
	ActiveMacTime   int64              `bson:"active_mac_time"`
	ActiveMacSource string             `bson:"active_mac_source"`
	IPs             []netutil.IP       `bson:"ips"`
	IPsTime         int64              `bson:"ips_time"`
	NetworkSwitch   netutil.SwitchName `bson:"network_switch"`
	NetworkPort     netutil.SwitchPort `bson:"network_port"`
	NetworkTime     int64              `bson:"network_timestamp"`
	NetworkSource   string             `bson:"network_source"`
}

type HostNetworkRepo struct {
	collection *mongo.Collection
}

func NewHostNetworkRepo(db *mongo.Database, pref *readpref.ReadPref) *HostNetworkRepo {
	return &HostNetworkRepo{
		collection: db.Collection(hostNetworkCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

func (repo *HostNetworkRepo) GetOrCreate(ctx context.Context, uuid string) (*HostNetwork, error) {
	opts := options.FindOneAndUpdate().SetUpsert(true).SetReturnDocument(options.After)
	result := &HostNetwork{}
	if err := repo.collection.FindOneAndUpdate(
		ctx,
		bson.D{{Key: HostNetworkFieldKeyID, Value: uuid}},
		bson.D{{Key: "$setOnInsert", Value: bson.D{{Key: HostNetworkFieldKeyID, Value: uuid}}}},
		opts,
	).Decode(result); err != nil && err != mongo.ErrNoDocuments {
		return nil, err
	}
	return result, nil
}

func (repo *HostMacsRepo) NewBulkWriter(opts *db.MongoBulkWriterOptions) (db.BulkWriter, error) {
	return db.NewMongoBulkWriter(repo.collection, opts)
}
