package repos

import (
	"context"
	"fmt"
	"sort"
	"testing"
	"time"

	"github.com/gofrs/uuid"
	"github.com/stretchr/testify/suite"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
)

type HostTestSuite struct {
	suite.Suite
	repo  *HostRepo
	hosts []*Host
}

func (suite *HostTestSuite) SetupSuite() {
	mongodb, err := db.GetTestingMongoDB()
	suite.Require().NoError(err)
	suite.repo = NewHostRepo(mongodb, readpref.Primary())
	var hosts []*Host
	for i := 0; i < 4; i++ {
		hosts = append(hosts, &Host{
			UUID:    HostUUID(uuid.Must(uuid.NewV4()).String()),
			Inv:     HostInv(i),
			Name:    HostName(fmt.Sprintf("host-%d", i)),
			Project: ProjectID(fmt.Sprintf("project-%d", i%3)),
			State:   fmt.Sprintf("state-%d", i%2),
			Status:  fmt.Sprintf("status-%d", i%2),
		})
	}
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	for _, host := range hosts {
		suite.Require().NoError(suite.repo.Insert(ctx, host))
	}
	suite.hosts = hosts
}

func (suite *HostTestSuite) TestFind() {
	type testcase struct {
		filter   *HostFilter
		expected []*Host
	}
	cases := []*testcase{
		{
			filter:   &HostFilter{Project: "project-0"},
			expected: []*Host{suite.hosts[0], suite.hosts[3]},
		},
		{
			filter:   &HostFilter{Project: "project-404"},
			expected: nil,
		},
		{
			filter:   nil,
			expected: suite.hosts,
		},
	}
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	for _, c := range cases {
		founded, err := suite.repo.Find(ctx, c.filter)
		suite.Require().NoError(err)
		suite.Equal(c.expected, founded)
	}
}

func (suite *HostTestSuite) TestFindCommon() {
	type testcase struct {
		filter   *HostFilter
		expected []*HostCommon
	}
	fullToCommon := func(h *Host) *HostCommon {
		return &HostCommon{
			Name:   h.Name,
			State:  h.State,
			Status: h.Status,
		}
	}
	cases := []*testcase{
		{
			filter:   &HostFilter{Project: "project-0"},
			expected: []*HostCommon{fullToCommon(suite.hosts[0]), fullToCommon(suite.hosts[3])},
		},
		{
			filter:   &HostFilter{Project: "project-404"},
			expected: nil,
		},
	}
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	for _, c := range cases {
		founded, err := suite.repo.FindCommon(ctx, c.filter)
		suite.Require().NoError(err)
		suite.Equal(c.expected, founded)
	}
}

func (suite *HostTestSuite) TestSelect() {
	type testcase struct {
		filter   *HostFilter
		keys     []string
		expected []*Host
	}
	cases := []*testcase{
		{
			filter: &HostFilter{Project: "project-0"},
			keys:   []string{HostFieldKeyName, HostFieldKeyStatus},
			expected: []*Host{
				{
					Name:   suite.hosts[0].Name,
					Status: suite.hosts[0].Status,
				},
				{
					Name:   suite.hosts[3].Name,
					Status: suite.hosts[3].Status,
				},
			},
		},
		{
			filter:   &HostFilter{Project: "project-404"},
			keys:     []string{HostFieldKeyName, HostFieldKeyStatus},
			expected: nil,
		},
	}
	reorderedKeys := &testcase{
		filter: &HostFilter{Project: "project-0"},
		keys:   []string{HostFieldKeyStatus, HostFieldKeyInv},
		expected: []*Host{
			{
				Inv:    suite.hosts[0].Inv,
				Status: suite.hosts[0].Status,
			},
			{
				Inv:    suite.hosts[3].Inv,
				Status: suite.hosts[3].Status,
			},
		},
	}
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	for _, c := range cases {
		res, err := suite.repo.Select(ctx, c.filter, c.keys)
		suite.Require().NoError(err)
		var founded []*Host
		for res.Next() {
			var host Host
			suite.Require().NoError(res.Scan(&host.Name, &host.Status))
			founded = append(founded, &host)
		}
		sort.Slice(founded, func(i, j int) bool { return founded[i].Name < founded[j].Name })
		suite.Equal(c.expected, founded)
	}
	res, err := suite.repo.Select(ctx, reorderedKeys.filter, reorderedKeys.keys)
	suite.Require().NoError(err)
	var founded []*Host
	for res.Next() {
		var host Host
		suite.Require().NoError(res.Scan(&host.Status, &host.Inv))
		founded = append(founded, &host)
		sort.Slice(founded, func(i, j int) bool { return founded[i].Inv < founded[j].Inv })

	}
	suite.Equal(reorderedKeys.expected, founded)
}

func TestHostRepo(t *testing.T) {
	suite.Run(t, new(HostTestSuite))
}
