package repos

import (
	"context"
	"encoding/json"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
)

const metricsCollectionName = "metrics"

const (
	MetricsFieldKeyID          = "_id"
	MetricsFieldKeyTime        = "time"
	MetricsFieldKeyYasmData    = "yasm_data"
	MetricsFieldKeySolomonData = "solomon_data"
)

type MetricGroup struct {
	ID          string                 `bson:"_id"`
	Time        int64                  `bson:"time"`
	YasmData    []interface{}          `bson:"yasm_data"`
	SolomonData map[string]interface{} `bson:"solomon_data"`
}

type MetricRepo struct {
	collection *mongo.Collection
}

func NewMetricRepo(db *mongo.Database, pref *readpref.ReadPref) *MetricRepo {
	return &MetricRepo{
		collection: db.Collection(metricsCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

func (repo *MetricRepo) GetData(ctx context.Context, filter *MetricFilter, dataField string) ([]byte, error) {
	filters := bson.D{}
	if filter != nil {
		filters = filter.getBSON()
	}
	selection, err := db.NewMongoSelection(ctx, repo.collection, filters, options.Find(), []string{dataField})
	if err != nil {
		return nil, err
	}
	var data map[string]interface{}
	if selection.Next() {
		if err = selection.Scan(&data); err != nil {
			return nil, err
		}
	}
	return json.Marshal(data)
}

func (repo *MetricRepo) Update(ctx context.Context, group *MetricGroup) error {
	opts := options.Update().SetUpsert(true)
	filter := &MetricFilter{
		ID:         group.ID,
		TimeBefore: group.Time,
	}
	filters := filter.getBSON()
	update := bson.D{{
		Key: "$set",
		Value: bson.M{
			MetricsFieldKeyTime:        group.Time,
			MetricsFieldKeyYasmData:    group.YasmData,
			MetricsFieldKeySolomonData: group.SolomonData,
		},
	}}
	_, err := repo.collection.UpdateOne(ctx, filters, update, opts)
	return err
}

type MetricFilter struct {
	ID         string
	TimeBefore int64
	TimeAfter  int64
}

func (f *MetricFilter) getBSON() bson.D {
	filters := bson.D{}
	if f.ID != "" {
		filters = append(filters, bson.E{Key: MetricsFieldKeyID, Value: f.ID})
	}
	if f.TimeBefore != 0 {
		filters = append(filters, bson.E{Key: MetricsFieldKeyTime, Value: bson.D{{Key: "$lt", Value: f.TimeBefore}}})
	}
	if f.TimeAfter != 0 {
		filters = append(filters, bson.E{Key: MetricsFieldKeyTime, Value: bson.D{{Key: "$gt", Value: f.TimeAfter}}})
	}
	return filters
}
