package repos

import (
	"context"
	"encoding/json"
	"testing"
	"time"

	"github.com/stretchr/testify/suite"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
)

type MetricTestSuite struct {
	suite.Suite
	repo   *MetricRepo
	groups []*MetricGroup
}

func (suite *MetricTestSuite) SetupSuite() {
	mongodb, err := db.GetTestingMongoDB()
	suite.Require().NoError(err)
	suite.repo = NewMetricRepo(mongodb, readpref.Primary())
}

func (suite *MetricTestSuite) TestUpdate() {
	t := time.Now().Unix()
	groups := []*MetricGroup{
		{
			ID:   "id",
			Time: t,
		},
		{
			ID:   "id",
			Time: t + 1,
		},
	}

	suite.Require().NoError(json.Unmarshal(
		[]byte(`[["m1",[[0,1],[10,2]]],["m2",[[0,2],[10,2]]],["m3",21]]`),
		&groups[0].YasmData,
	))
	suite.Require().NoError(json.Unmarshal(
		[]byte(`{"metrics":[{"labels":{"name":"m1"},"value":1},{"labels":{"name":"m2"},"value":2}]}`),
		&groups[0].SolomonData,
	))

	updatedYasm := []byte(`[["m1",[[0,1],[10,2]]],["m2",[[0,2],[10,12]]],["m4",25]]`)
	updatedSolomon := []byte(`{"metrics":[{"labels":{"name":"m1"},"value":1},{"labels":{"name":"m2"},"value":20}]}`)
	suite.Require().NoError(json.Unmarshal(updatedYasm, &groups[1].YasmData))
	suite.Require().NoError(json.Unmarshal(updatedSolomon, &groups[1].SolomonData))

	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	_, err := suite.repo.collection.InsertOne(ctx, groups[0])
	suite.Require().NoError(err)
	suite.Require().NoError(suite.repo.Update(ctx, groups[1]))
	opts := options.Find()
	cursor, err := suite.repo.collection.Find(ctx, bson.D{}, opts)
	suite.Require().NoError(err)
	var results []MetricGroup
	suite.Require().NoError(cursor.All(ctx, &results))

	gotUpdatedYasm, err := json.Marshal(results[0].YasmData)
	suite.Require().NoError(err)
	gotUpdatedSolomon, err := json.Marshal(results[0].SolomonData)
	suite.Require().NoError(err)

	suite.Require().Equal(updatedYasm, gotUpdatedYasm)
	suite.Require().Equal(updatedSolomon, gotUpdatedSolomon)
}

func TestMetricRepo(t *testing.T) {
	suite.Run(t, new(MetricTestSuite))
}
