package repos

import (
	"context"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
)

const projectCollectionName = "projects"

const (
	ProjectFieldKeyID                = "_id"
	ProjectFieldKeyName              = "name"
	ProjectFieldKeyTags              = "tags"
	ProjectFieldKeyDNSAutomation     = "dns_automation"
	ProjectFieldKeyHealingAutomation = "healing_automation"
)

type ProjectID string

type Project struct {
	ID   ProjectID `bson:"_id"`
	Name string    `bson:"name"`
	Tags []string  `bson:"tags,omitempty"`
}

type ProjectCommon struct {
	ID   ProjectID
	Tags []string
}

func NewProjectRepo(db *mongo.Database, pref *readpref.ReadPref) *ProjectRepo {
	return &ProjectRepo{
		collection: db.Collection(projectCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

type ProjectRepo struct {
	collection *mongo.Collection
}

func (repo *ProjectRepo) Find(ctx context.Context, filter *ProjectFilter) ([]*Project, error) {
	opts := options.Find()
	filters := filter.getBSON()
	var projects []*Project
	cursor, err := repo.collection.Find(ctx, filters, opts)
	if err != nil {
		return nil, err

	}
	if err := cursor.All(ctx, &projects); err != nil {
		return nil, err
	}
	return projects, nil
}

func (repo *ProjectRepo) FindCommon(ctx context.Context, filter *ProjectFilter) ([]*ProjectCommon, error) {
	selection, err := repo.Select(ctx, filter, []string{ProjectFieldKeyID, ProjectFieldKeyTags})
	if err != nil {
		return nil, err
	}
	var projects []*ProjectCommon
	for selection.Next() {
		project := &ProjectCommon{}
		if err = selection.Scan(&project.ID, &project.Tags); err != nil {
			return nil, err
		}
		projects = append(projects, project)
	}
	return projects, nil
}

func (repo *ProjectRepo) FindFieldValues(
	ctx context.Context,
	filter *ProjectFilter,
	fieldKey string,
) ([]string, error) {
	selection, err := repo.Select(ctx, filter, []string{fieldKey})
	if err != nil {
		return nil, err
	}
	var fields []string
	for selection.Next() {
		var field string
		if err = selection.Scan(&field); err != nil {
			return nil, err
		}
		fields = append(fields, field)
	}
	return fields, nil
}

func (repo *ProjectRepo) Insert(ctx context.Context, project *Project) error {
	_, err := repo.collection.InsertOne(ctx, project)
	return err
}

func (repo *ProjectRepo) Select(ctx context.Context, filter *ProjectFilter, keys []string) (*db.MongoSelection, error) {
	filters := bson.D{}
	if filter != nil {
		filters = filter.getBSON()
	}
	return db.NewMongoSelection(ctx, repo.collection, filters, options.Find(), keys)
}

type ProjectFilter struct {
	Automated bool
	RTC       bool
}

func (f *ProjectFilter) getBSON() bson.D {
	filters := bson.D{}
	if f.Automated {
		dns := bson.M{ProjectFieldKeyDNSAutomation + ".enabled": true}
		healing := bson.M{ProjectFieldKeyHealingAutomation + ".enabled": true}
		filters = append(filters, bson.E{Key: "$or", Value: bson.A{dns, healing}})
	}
	if f.RTC {
		filters = append(filters, bson.E{Key: ProjectFieldKeyTags, Value: "rtc"})
	}
	return filters
}
