package repos

import (
	"context"
	"fmt"
	"testing"
	"time"

	"github.com/stretchr/testify/suite"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo/readpref"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
)

type ProjectTestSuite struct {
	suite.Suite
	repo     *ProjectRepo
	projects []*Project
}

func (suite *ProjectTestSuite) SetupSuite() {
	mongodb, err := db.GetTestingMongoDB()
	suite.Require().NoError(err)
	suite.repo = NewProjectRepo(mongodb, readpref.Primary())
	var projects []*Project
	for i := 0; i < 2; i++ {
		projects = append(projects, &Project{
			ID:   ProjectID(fmt.Sprintf("id-%d", i)),
			Name: fmt.Sprintf("name-%d", i),
			Tags: []string{"tag-1", "tag-2"},
		})
	}
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	for _, project := range projects {
		suite.Require().NoError(suite.repo.Insert(ctx, project))
	}
	projects = append(projects, &Project{
		ID:   "id-2",
		Name: "name",
	})
	_, err = suite.repo.collection.InsertOne(
		ctx,
		bson.D{{Key: "_id", Value: projects[2].ID}, {Key: "name", Value: projects[2].Name}},
	)
	suite.Require().NoError(err)
	suite.projects = projects
}

func (suite *ProjectTestSuite) TestFindCommon() {
	type testcase struct {
		filter   *ProjectFilter
		expected []*ProjectCommon
	}
	fullToCommon := func(p *Project) *ProjectCommon {
		return &ProjectCommon{
			ID:   p.ID,
			Tags: p.Tags,
		}
	}
	cases := []*testcase{
		{
			filter: &ProjectFilter{},
			expected: []*ProjectCommon{
				fullToCommon(suite.projects[0]),
				fullToCommon(suite.projects[1]),
				fullToCommon(suite.projects[2]),
			},
		},
	}
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	for _, c := range cases {
		founded, err := suite.repo.FindCommon(ctx, c.filter)
		suite.Require().NoError(err)
		suite.Equal(c.expected, founded)
	}
}

func (suite *ProjectTestSuite) TestFindStringFieldValues() {
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	founded, err := suite.repo.FindFieldValues(ctx, &ProjectFilter{}, ProjectFieldKeyName)
	suite.Require().NoError(err)
	suite.Equal(founded, []string{suite.projects[0].Name, suite.projects[1].Name, suite.projects[2].Name})
}

func (suite *ProjectTestSuite) TestFindProjectIDFieldValues() {
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()
	founded, err := suite.repo.FindFieldValues(ctx, &ProjectFilter{}, ProjectFieldKeyID)
	suite.Require().NoError(err)
	suite.Equal(founded, []string{
		string(suite.projects[0].ID),
		string(suite.projects[1].ID),
		string(suite.projects[2].ID),
	})
}

func TestProjectRepo(t *testing.T) {
	suite.Run(t, new(ProjectTestSuite))
}
