package repos

import (
	"context"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/readpref"
)

const shardCollectionName = "host_shard"

const (
	ShardFieldKeyType = "type"

	ShardTypeScreening = "screening"
)

type Shard struct {
	UUID           string `bson:"_id"`
	ProcessingTime int64  `bson:"processing_time"`
	ID             int    `bson:"shard_id"`
	Tier           string `bson:"tier"`
	Type           string `bson:"type"`
}

type ShardRepo struct {
	collection *mongo.Collection
}

func NewShardRepo(db *mongo.Database, pref *readpref.ReadPref) *ShardRepo {
	return &ShardRepo{
		collection: db.Collection(shardCollectionName, options.Collection().SetReadPreference(pref)),
	}
}

func (repo *ShardRepo) Find(ctx context.Context, filter *ShardFilter) ([]*Shard, error) {
	filters := bson.D{}
	if filter != nil {
		filters = filter.getBSON()
	}
	cursor, err := repo.collection.Find(ctx, filters, options.Find())
	if err != nil {
		return nil, err
	}
	var shards []*Shard
	if err := cursor.All(ctx, &shards); err != nil {
		return nil, err
	}
	return shards, nil
}

type ShardFilter struct {
	Type string
}

func (f *ShardFilter) getBSON() bson.D {
	filters := bson.D{}
	if f.Type != "" {
		filters = append(filters, bson.E{Key: ShardFieldKeyType, Value: f.Type})
	}
	return filters
}
